<?php
/**
 * Azo Widget Recent Posts
 *
 * @package azo
 */

/**
 * Azo Widget Recent Posts Thumbnails.
 */
class Azo_Widget_Recent_Posts extends WP_Widget {
	/**
	 * Set up a new Recent Posts with Thumbnails widget instance.
	 */
	public function __construct() {
		$widget_ops = array(
			'classname'                   => 'azo_widget_recent_entries',
			'description'                 => 'Display most recent posts with thumbnails.',
			'customize_selective_refresh' => true,
		);
		parent::__construct( 'azo-recent-posts', esc_html__( 'Recent Posts with Thumbnails', 'azo' ), $widget_ops );
	}

	/**
	 * Output the content for the current Recent Posts with Thumbnails widget instance.
	 *
	 * @param string $args type.
	 * @param string $instance type.
	 */
	public function widget( $args, $instance ) {
		if ( ! isset( $args['widget_id'] ) ) {
			$args['widget_id'] = $this->id;
		}

		$title = ( ! empty( $instance['title'] ) ) ? $instance['title'] : esc_html__( 'Recent Posts', 'azo' );
		$title = apply_filters( 'widget_title', $title, $instance, $this->id_base );

		$number = ( ! empty( $instance['number'] ) ) ? absint( $instance['number'] ) : 3;
		if ( ! $number ) {
			$number = 3;
		}

		$recent_posts = new WP_Query(
			array(
				'posts_per_page'      => $number,
				'post_status'         => 'publish',
				'no_found_rows'       => true,
				'ignore_sticky_posts' => true,
			)
		);

		if ( ! $recent_posts->have_posts() ) {
			return;
		}

		echo wp_kses_post( $args['before_widget'] );

		if ( $title ) {
			echo wp_kses_post( $args['before_title'] . $title . $args['after_title'] );
		}
		?>

		<ul class="azo-entries">
			<?php
			while ( $recent_posts->have_posts() ) :
				$recent_posts->the_post();
				$post_title = get_the_title();
				$title      = ! empty( $post_title ) ? $post_title : esc_html__( '(no title)', 'azo' );
				?>
				<li class="azo-entry">
					<?php if ( has_post_thumbnail() ) : ?>
						<div class="azo-entry-thumbnail">
							<a href="<?php the_permalink(); ?>" aria-hidden="true" tabindex="-1">
								<?php the_post_thumbnail( 'thumbnail' ); ?>
							</a>
						</div>
					<?php endif; ?>

					<div class="azo-recent-post-detail">
						<a href="<?php the_permalink(); ?>" class="azo-recent-post-title"><?php echo esc_html( $title ); ?></a>
						<span class="azo-recent-post-on">
							<?php echo get_the_date( get_option( 'date_format' ) ); ?>
						</span>
					</div>
				</li><!-- .azo-entry -->
				<?php
			endwhile;
			wp_reset_postdata();
			?>
		</ul><!-- .azo-entries -->

		<?php
		echo wp_kses_post( $args['after_widget'] );
	}

	/**
	 * Handles updating the settings for the current Recent Posts with Thumbnails widget instance.
	 *
	 * @param array $new_instance type.
	 * @param array $old_instance type.
	 *
	 * @return array
	 */
	public function update( $new_instance, $old_instance ) {
		$instance           = $old_instance;
		$instance['title']  = sanitize_text_field( $new_instance['title'] );
		$instance['number'] = (int) $new_instance['number'];

		return $instance;
	}

	/**
	 * Outputs the settings form for the azo Recent Posts with Thumbnails widget.
	 *
	 * @param array $instance type.
	 * @return string|void type.
	 */
	public function form( $instance ) {
		$title  = isset( $instance['title'] ) ? esc_attr( $instance['title'] ) : '';
		$number = isset( $instance['number'] ) ? absint( $instance['number'] ) : 3;
		?>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"><?php esc_html_e( 'Title', 'azo' ); ?></label>
			<input
				type="text"
				class="widefat"
				id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"
				name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>"
				value="<?php echo esc_attr( $title ); ?>">
		</p>

		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'number' ) ); ?>"><?php esc_html_e( 'Number of posts to show:', 'azo' ); ?></label>
			<input
				type="number"
				step="1"
				min="1"
				size="3"
				class="tiny-text"
				id="<?php echo esc_attr( $this->get_field_id( 'number' ) ); ?>"
				name="<?php echo esc_attr( $this->get_field_name( 'number' ) ); ?>"
				value="<?php echo esc_attr( $number ); ?>">
		</p>
		<?php
	}
}

