<?php
/**
 * Product Filter
 *
 * @package azo
 */

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'Azo_Widget_Product_Filter' ) ) :
	/**
	 * Class Azo_Widget_Product_Filter
	 */
	class Azo_Widget_Product_Filter extends WC_Widget  { // @codingStandardsIgnoreLine
		/**
		 * Constructor.
		 */
		public function __construct() {
			$this->widget_cssclass    = 'woocommerce widget_layered_nav woocommerce-widget-layered-nav';
			$this->widget_description = __( 'Display a list of attributes to filter products in your store.', 'azo' );
			$this->widget_id          = 'azo_woocommerce_layered_nav';
			$this->widget_name        = __( 'AZO - Filter Products by Attribute', 'azo' );
			parent::__construct();
		}

		/**
		 * Updates a particular instance of a widget.
		 *
		 * @see WP_Widget->update
		 *
		 * @param array $new_instance New Instance.
		 * @param array $old_instance Old Instance.
		 *
		 * @return array
		 */
		public function update( $new_instance, $old_instance ) {
			$this->init_settings();
			return parent::update( $new_instance, $old_instance );
		}

		/**
		 * Outputs the settings update form.
		 *
		 * @see WP_Widget->form
		 *
		 * @param array $instance Instance.
		 */
		public function form( $instance ) {
			$this->init_settings();
			parent::form( $instance );
		}

		/**
		 * Init settings after post types are registered.
		 */
		public function init_settings() {
			$attribute_array      = array();
			$std_attribute        = '';
			$attribute_taxonomies = wc_get_attribute_taxonomies();

			if ( ! empty( $attribute_taxonomies ) ) {
				foreach ( $attribute_taxonomies as $tax ) {
					if ( taxonomy_exists( wc_attribute_taxonomy_name( $tax->attribute_name ) ) ) {
						$attribute_array[ $tax->attribute_name ] = $tax->attribute_name;
					}
				}
				$std_attribute = current( $attribute_array );
			}

			$this->settings = array(
				'title'              => array(
					'type'  => 'text',
					'std'   => __( 'Filter by', 'azo' ),
					'label' => __( 'Title', 'azo' ),
				),
				'attribute'          => array(
					'type'    => 'select',
					'std'     => $std_attribute,
					'label'   => __( 'Attribute', 'azo' ),
					'options' => $attribute_array,
				),
				'display_type'       => array(
					'type'    => 'select',
					'std'     => 'list',
					'label'   => __( 'Display type', 'azo' ),
					'options' => array(
						'list'     => __( 'List', 'azo' ),
						'dropdown' => __( 'Dropdown', 'azo' ),
					),
				),
				'query_type'         => array(
					'type'    => 'select',
					'std'     => 'and',
					'label'   => __( 'Query type', 'azo' ),
					'options' => array(
						'and' => __( 'AND', 'azo' ),
						'or'  => __( 'OR', 'azo' ),
					),
				),
				'show_product_count' => array(
					'type'  => 'checkbox',
					'std'   => 1,
					'label' => __( 'Show product count', 'azo' ),
				),
			);
		}

		/**
		 * Get this widgets taxonomy.
		 *
		 * @param array $instance Array of instance options.
		 * @return string
		 */
		protected function get_instance_taxonomy( $instance ) {
			if ( isset( $instance['attribute'] ) ) {
				return wc_attribute_taxonomy_name( $instance['attribute'] );
			}

			$attribute_taxonomies = wc_get_attribute_taxonomies();

			if ( ! empty( $attribute_taxonomies ) ) {
				foreach ( $attribute_taxonomies as $tax ) {
					if ( taxonomy_exists( wc_attribute_taxonomy_name( $tax->attribute_name ) ) ) {
						return wc_attribute_taxonomy_name( $tax->attribute_name );
					}
				}
			}

			return '';
		}

		/**
		 * Get this widgets query type.
		 *
		 * @param array $instance Array of instance options.
		 * @return string
		 */
		protected function get_instance_query_type( $instance ) {
			return isset( $instance['query_type'] ) ? $instance['query_type'] : 'and';
		}

		/**
		 * Get this widgets display type.
		 *
		 * @param array $instance Array of instance options.
		 * @return string
		 */
		protected function get_instance_display_type( $instance ) {
			return isset( $instance['display_type'] ) ? $instance['display_type'] : 'list';
		}

		/**
		 * Get this widgets show product count.
		 *
		 * @param array $instance Array of instance options.
		 * @return string
		 */
		protected function get_instance_show_product_count( $instance ) {
			return isset( $instance['show_product_count'] ) ? $instance['show_product_count'] : 1;
		}

		/**
		 * Output widget.
		 *
		 * @see WP_Widget
		 *
		 * @param array $args Arguments.
		 * @param array $instance Instance.
		 */
		public function widget( $args, $instance ) {
			if ( ! is_shop() && ! is_product_taxonomy() ) {
				return;
			}

			$_chosen_attributes = WC_Query::get_layered_nav_chosen_attributes();
			$taxonomy           = $this->get_instance_taxonomy( $instance );
			$query_type         = $this->get_instance_query_type( $instance );
			$display_type       = $this->get_instance_display_type( $instance );
			$show_product_count = $this->get_instance_show_product_count( $instance );

			if ( ! taxonomy_exists( $taxonomy ) ) {
				return;
			}

			$terms = get_terms( $taxonomy, array( 'hide_empty' => '1' ) );

			if ( 0 === count( $terms ) ) {
				return;
			}

			ob_start();

			$this->widget_start( $args, $instance );

			if ( 'dropdown' === $display_type ) {
				wp_enqueue_script( 'selectWoo' );
				wp_enqueue_style( 'select2' );
				$found = $this->layered_nav_dropdown( $terms, $taxonomy, $query_type );
			} else {
				$found = $this->layered_nav_list( $terms, $taxonomy, $query_type, $show_product_count );
			}

			$this->widget_end( $args );

			// Force found when option is selected - do not force found on taxonomy attributes.
			if ( ! is_tax() && is_array( $_chosen_attributes ) && array_key_exists( $taxonomy, $_chosen_attributes ) ) {
				$found = true;
			}

			if ( ! $found ) {
				ob_end_clean();
			} else {
				echo ob_get_clean(); // @codingStandardsIgnoreLine
			}
		}

		/**
		 * Return the currently viewed taxonomy name.
		 *
		 * @return string
		 */
		protected function get_current_taxonomy() {
			return is_tax() ? get_queried_object()->taxonomy : '';
		}

		/**
		 * Return the currently viewed term ID.
		 *
		 * @return int
		 */
		protected function get_current_term_id() {
			return absint( is_tax() ? get_queried_object()->term_id : 0 );
		}

		/**
		 * Return the currently viewed term slug.
		 *
		 * @return int
		 */
		protected function get_current_term_slug() {
			return absint( is_tax() ? get_queried_object()->slug : 0 );
		}

		/**
		 * Show dropdown layered nav.
		 *
		 * @param  array  $terms Terms.
		 * @param  string $taxonomy Taxonomy.
		 * @param  string $query_type Query Type.
		 * @return bool Will nav display?
		 */
		protected function layered_nav_dropdown( $terms, $taxonomy, $query_type ) {
			global $wp;
			$found = false;

			if ( $taxonomy !== $this->get_current_taxonomy() ) {
				$term_counts          = $this->get_filtered_term_product_counts( wp_list_pluck( $terms, 'term_id' ), $taxonomy, $query_type );
				$_chosen_attributes   = WC_Query::get_layered_nav_chosen_attributes();
				$taxonomy_filter_name = wc_attribute_taxonomy_slug( $taxonomy );
				$taxonomy_label       = wc_attribute_label( $taxonomy );

				/* translators: %s: taxonomy name */
				$any_label      = apply_filters( 'woocommerce_layered_nav_any_label', sprintf( __( 'Any %s', 'azo' ), $taxonomy_label ), $taxonomy_label, $taxonomy );
				$multiple       = 'or' === $query_type;
				$current_values = isset( $_chosen_attributes[ $taxonomy ]['terms'] ) ? $_chosen_attributes[ $taxonomy ]['terms'] : array();

				if ( '' === get_option( 'permalink_structure' ) ) {
					$form_action = remove_query_arg( array( 'page', 'paged' ), add_query_arg( $wp->query_string, '', home_url( $wp->request ) ) );
				} else {
					$form_action = preg_replace( '%\/page/[0-9]+%', '', home_url( user_trailingslashit( $wp->request ) ) );
				}

				echo '<form method="get" action="' . esc_url( $form_action ) . '" class="woocommerce-widget-layered-nav-dropdown">';
				echo '<select class="woocommerce-widget-layered-nav-dropdown dropdown_layered_nav_' . esc_attr( $taxonomy_filter_name ) . '"' . ( $multiple ? 'multiple="multiple"' : '' ) . '>';
				echo '<option value="">' . esc_html( $any_label ) . '</option>';

				foreach ( $terms as $term ) {

					// If on a term page, skip that term in widget list.
					if ( $term->term_id === $this->get_current_term_id() ) {
						continue;
					}

					// Get count based on current view.
					$option_is_set = in_array( $term->slug, $current_values, true );
					$count         = isset( $term_counts[ $term->term_id ] ) ? $term_counts[ $term->term_id ] : 0;

					// Only show options with count > 0.
					if ( 0 < $count ) {
						$found = true;
					} elseif ( 0 === $count && ! $option_is_set ) {
						continue;
					}

					echo '<option value="' . esc_attr( urldecode( $term->slug ) ) . '" ' . selected( $option_is_set, true, false ) . '>' . esc_html( $term->name ) . '</option>';
				}

				echo '</select>';

				if ( $multiple ) {
					echo '<button class="woocommerce-widget-layered-nav-dropdown__submit" type="submit" value="' . esc_attr__( 'Apply', 'azo' ) . '">' . esc_html__( 'Apply', 'azo' ) . '</button>';
				}

				if ( 'or' === $query_type ) {
					echo '<input type="hidden" name="query_type_' . esc_attr( $taxonomy_filter_name ) . '" value="or" />';
				}

				echo '<input type="hidden" name="filter_' . esc_attr( $taxonomy_filter_name ) . '" value="' . esc_attr( implode( ',', $current_values ) ) . '" />';
				echo wc_query_string_form_fields( null, array( 'filter_' . $taxonomy_filter_name, 'query_type_' . $taxonomy_filter_name ), '', true ); // @codingStandardsIgnoreLine
				echo '</form>';

				wc_enqueue_js(
					"
				// Update value on change.
				jQuery( '.dropdown_layered_nav_" . esc_js( $taxonomy_filter_name ) . "' ).on( 'change', function() {
					var slug = jQuery( this ).val();
					jQuery( ':input[name=\"filter_" . esc_js( $taxonomy_filter_name ) . "\"]' ).val( slug );
					// Submit form on change if standard dropdown.
					if ( ! jQuery( this ).attr( 'multiple' ) ) {
						jQuery( this ).closest( 'form' ).trigger( 'submit' );
					}
				});
				// Use Select2 enhancement if possible
				if ( jQuery().selectWoo ) {
					var wc_layered_nav_select = function() {
						jQuery( '.dropdown_layered_nav_" . esc_js( $taxonomy_filter_name ) . "' ).selectWoo( {
							placeholder: decodeURIComponent('" . rawurlencode( (string) wp_specialchars_decode( $any_label ) ) . "'),
							minimumResultsForSearch: 5,
							width: '100%',
							allowClear: " . ( $multiple ? 'false' : 'true' ) . ",
							language: {
								noResults: function() {
									return '" . esc_js( _x( 'No matches found', 'enhanced select', 'azo' ) ) . "';
								}
							}
						} );
					};
					wc_layered_nav_select();
				}
			"
				);
			}

			return $found;
		}

		/**
		 * Count products within certain terms, taking the main WP query into consideration.
		 *
		 * This query allows counts to be generated based on the viewed products, not all products.
		 *
		 * @param  array  $term_ids Term IDs.
		 * @param  string $taxonomy Taxonomy.
		 * @param  string $query_type Query Type.
		 * @return array
		 */
		protected function get_filtered_term_product_counts( $term_ids, $taxonomy, $query_type ) {
			global $wpdb;

			$tax_query  = $this->get_main_tax_query();
			$meta_query = $this->get_main_meta_query();

			if ( 'or' === $query_type ) {
				foreach ( $tax_query as $key => $query ) {
					if ( is_array( $query ) && $taxonomy === $query['taxonomy'] ) {
						unset( $tax_query[ $key ] );
					}
				}
			}

			$meta_query     = new WP_Meta_Query( $meta_query );
			$tax_query      = new WP_Tax_Query( $tax_query );
			$meta_query_sql = $meta_query->get_sql( 'post', $wpdb->posts, 'ID' );
			$tax_query_sql  = $tax_query->get_sql( $wpdb->posts, 'ID' );
			$term_ids_sql   = '(' . implode( ',', array_map( 'absint', $term_ids ) ) . ')';

			// Generate query.
			$query           = array();
			$query['select'] = "SELECT COUNT( DISTINCT {$wpdb->posts}.ID ) AS term_count, terms.term_id AS term_count_id";
			$query['from']   = "FROM {$wpdb->posts}";
			$query['join']   = "
			INNER JOIN {$wpdb->term_relationships} AS term_relationships ON {$wpdb->posts}.ID = term_relationships.object_id
			INNER JOIN {$wpdb->term_taxonomy} AS term_taxonomy USING( term_taxonomy_id )
			INNER JOIN {$wpdb->terms} AS terms USING( term_id )
			" . $tax_query_sql['join'] . $meta_query_sql['join'];

			$query['where'] = "
			WHERE {$wpdb->posts}.post_type IN ( 'product' )
			AND {$wpdb->posts}.post_status = 'publish'
			{$tax_query_sql['where']} {$meta_query_sql['where']}
			AND terms.term_id IN $term_ids_sql";

			$search = $this->get_main_search_query_sql();
			if ( $search ) {
				$query['where'] .= ' AND ' . $search;
			}

			$query['group_by'] = 'GROUP BY terms.term_id';
			$query             = apply_filters( 'woocommerce_get_filtered_term_product_counts_query', $query );
			$query_sql         = implode( ' ', $query );

			// We have a query - let's see if cached results of this query already exist.
			$query_hash = md5( $query_sql );

			// Maybe store a transient of the count values.
			$cache = apply_filters( 'woocommerce_layered_nav_count_maybe_cache', true );
			if ( true === $cache ) {
				$cached_counts = (array) get_transient( 'wc_layered_nav_counts_' . sanitize_title( $taxonomy ) );
			} else {
				$cached_counts = array();
			}

			if ( ! isset( $cached_counts[ $query_hash ] ) ) {
				// phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
				$results                      = $wpdb->get_results( $query_sql, ARRAY_A );
				$counts                       = array_map( 'absint', wp_list_pluck( $results, 'term_count', 'term_count_id' ) );
				$cached_counts[ $query_hash ] = $counts;
				if ( true === $cache ) {
					set_transient( 'wc_layered_nav_counts_' . sanitize_title( $taxonomy ), $cached_counts, DAY_IN_SECONDS );
				}
			}

			return array_map( 'absint', (array) $cached_counts[ $query_hash ] );
		}

		/**
		 * Wrapper for WC_Query::get_main_tax_query() to ease unit testing.
		 *
		 * @since 4.4.0
		 * @return array
		 */
		protected function get_main_tax_query() {
			return WC_Query::get_main_tax_query();
		}

		/**
		 * Wrapper for WC_Query::get_main_search_query_sql() to ease unit testing.
		 *
		 * @since 4.4.0
		 * @return string
		 */
		protected function get_main_search_query_sql() {
			return WC_Query::get_main_search_query_sql();
		}

		/**
		 * Wrapper for WC_Query::get_main_search_queryget_main_meta_query to ease unit testing.
		 *
		 * @since 4.4.0
		 * @return array
		 */
		protected function get_main_meta_query() {
			return WC_Query::get_main_meta_query();
		}

		/**
		 * Show list based layered nav.
		 *
		 * @param  array  $terms Terms.
		 * @param  string $taxonomy Taxonomy.
		 * @param  string $query_type Query Type.
		 * @return bool   Will nav display?
		 */
		protected function layered_nav_list( $terms, $taxonomy, $query_type, $show_product_count ) {
			// Attribute type: color || image || button.
			// Use new  Color Swatches plugin: https://wordpress.org/plugins/woo-variation-swatches/.
			if ( ! function_exists( 'Woo_Variation_Swatches' ) ) {
				return;
			}
			$attribute = wvs_get_wc_attribute_taxonomy( $taxonomy );

			$attribute_name = $attribute->attribute_name;
			$attribute_type = $attribute->attribute_type;
			$filter_name    = 'filter_' . $attribute_name;

			// List display.
			echo '<ul class="widget-products-filter filter-by-' . esc_attr( $attribute_type ) . '">';

			$term_counts        = $this->get_filtered_term_product_counts( wp_list_pluck( $terms, 'term_id' ), $taxonomy, $query_type );
			$_chosen_attributes = WC_Query::get_layered_nav_chosen_attributes();
			$found              = false;

			foreach ( $terms as $term ) {
				$current_values = isset( $_chosen_attributes[ $taxonomy ]['terms'] ) ? $_chosen_attributes[ $taxonomy ]['terms'] : array();
				$option_is_set  = in_array( $term->slug, $current_values, true );
				$count          = isset( $term_counts[ $term->term_id ] ) ? $term_counts[ $term->term_id ] : 0;

				// Skip the term for the current archive.
				if ( $this->get_current_term_id() === $term->term_id ) {
					continue;
				}

				// Only show options with count > 0.
				if ( 0 < $count ) {
					$found = true;
				} elseif ( 0 === $count && ! $option_is_set ) {
					continue;
				}
				$current_filter = isset( $_GET[ $filter_name ] ) ? explode( ',', wc_clean( wp_unslash( $_GET[ $filter_name ] ) ) ) : array(); // @codingStandardsIgnoreLine
				$current_filter = array_map( 'sanitize_title', $current_filter );

				if ( ! in_array( $term->slug, $current_filter, true ) ) {
					$current_filter[] = $term->slug;
				}

				$link = remove_query_arg( $filter_name, $this->get_current_page_url() );

				// Add current filters to URL.
				foreach ( $current_filter as $key => $value ) {
					// Exclude query arg for current term archive term.
					if ( $value === $this->get_current_term_slug() ) {
						unset( $current_filter[ $key ] );
					}

					// Exclude self so filter can be unset on click.
					if ( $option_is_set && $value === $term->slug ) {
						unset( $current_filter[ $key ] );
					}
				}

				if ( ! empty( $current_filter ) ) {
					asort( $current_filter );
					$link = add_query_arg( $filter_name, implode( ',', $current_filter ), $link );

					// Add Query type Arg to URL.
					if ( 'or' === $query_type && ! ( 1 === count( $current_filter ) && $option_is_set ) ) {
						$link = add_query_arg( 'query_type_' . sanitize_title( str_replace( 'pa_', '', $taxonomy ) ), 'or', $link );
					}
					$link = str_replace( '%2C', ',', $link );
				}

				// Term meta.
				$term_meta = get_term_meta( $term->term_id, 'product_attribute_' . $attribute_type, true );
				$image     = '';

				$output = '';

				if ( 'color' === $attribute_type ) {
					$output = 'style="color: ' . $term_meta . '"';
				} elseif ( 'image' === $attribute_type ) {
					$image = $term_meta ? wp_get_attachment_image_src( $term_meta ) : '';
					$image = $image ? $image[0] : WC()->plugin_url() . '/assets/images/placeholder.png';
				}

				if ( $count > 0 || $option_is_set ) {
					$link      = esc_url( apply_filters( 'azo_layered_nav_link', $link, $term, $taxonomy ) );
					$term_html = '<a class="pf-link" rel="nofollow" href="' . $link . '"><span class="swatch-inner">';

					if ( 'color' === $attribute_type || 'button' === $attribute_type ) {
						$term_html .= '<span class="pf-swatch-filter"><span ' . $output . '>' . esc_html( $term->name ) . '</span></span>';
					}
					if ( 'image' === $attribute_type ) {
						$term_html .= '<span class="pf-swatch-filter"><span style="background-image: url(' . $image . ')">' . esc_html( $term->name ) . '</span></span>';
					}
					$term_html .= '<span class="pf-label">' . esc_html( $term->name ) . '</span>';

					$term_html .= '</span>';

					$term_html .= '</a>';
					if ( 1 === $show_product_count ) {
						$term_html .= '<span class="pf-count">' . esc_html( $count ) . '</span>';
					}
				} else {
					$link      = false;
					$term_html = '<span>' . esc_html( $term->name ) . '</span>';
				}

				echo '<li class="pf-item wc-layered-nav-term with-swatch-' . esc_attr( $attribute_name ) . ' ' . ( $option_is_set ? 'active' : '' ) . '">';
				echo wp_kses_post( apply_filters( 'azo_layered_nav_term_html', $term_html, $term, $link, $count ) );
				echo '</li>';
			}

			echo '</ul>';

			return $found;
		}
	}
endif;
