<?php
/**
 * Azo Shop_Reviews
 *
 * @package  azo
 */

defined( 'ABSPATH' ) || exit;

class Shop_Reviews {

	protected static $instance = null;

	public static function instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	public function initialize() {
		/**
		 * Add to single product tabs
		 */
		/*add_action( 'wp_enqueue_scripts', [ $this, 'frontend_scripts' ] );
		add_filter( 'azo/product/bottom_tabs', [ $this, 'add_shop_reviews_tab' ] );*/

		add_action( 'wp_ajax_azo_get_shop_reviews', [ $this, 'get_shop_reviews_via_ajax' ] );
		add_action( 'wp_ajax_nopriv_azo_get_shop_reviews', [ $this, 'get_shop_reviews_via_ajax' ] );
	}

	public function frontend_scripts() {
		wp_register_script( 'azo-woo-shop-reviews', AZO_URI . "/assets/js/woo/shop-reviews.js", [ 'jquery' ], AZO_VERSION, true );

		wp_enqueue_script( 'azo-woo-shop-reviews' );
	}

	public function add_shop_reviews_tab( $tabs ) {
		global $product;

		if ( comments_open() ) {
			$tabs['shop-reviews'] = array(
				'title'    => __( 'Shop Reviews', 'azo' ),
				'priority' => 5,
				'callback' => array( $this, 'output_shop_reviews_html' ),
			);
		}

		return $tabs;
	}

	public function get_shop_reviews_via_ajax() {
		$current_page = ! empty( $_GET['current_page'] ) ? intval( $_GET['current_page'] ) : 1;
		$sorted_by    = isset( $_GET['sorting'] ) ? \Azo_Helper::data_clean( $_GET['sorting'] ) : '';
		$search_term  = isset( $_GET['keyword'] ) ? \Azo_Helper::data_clean( $_GET['keyword'] ) : '';

		if ( empty( $sorted_by ) ) {
			$sorted_by = 'recent';
		}

		$total_reviews = $this->get_shop_reviews_count( $search_term );
		$per_page      = $this->get_shop_reviews_per_page();
		$total_pages   = ceil( $total_reviews / $per_page );

		if ( $current_page > $total_pages ) {
			$current_page = $total_pages;
		}

		$offset = ( $current_page - 1 ) * $per_page;

		$comments = $this->get_shop_reviews( $offset, $per_page, $search_term, $sorted_by );

		ob_start();
		wp_list_comments( array(
			'style'       => 'ol',
			'callback'    => array( '\Azo\Woo\Shop_Reviews', 'output_shop_review_html' ),
			'short_ping'  => true,
			'avatar_size' => 60,
		), $comments );
		$comment_html = ob_get_clean();

		ob_start();
		\Azo_Templates::render_paginate_links( [
			'format'  => '?current_page=%#%',
			'current' => max( 1, $current_page ),
			'total'   => $total_pages,
		] );
		$pagination_html = ob_get_clean();

		$fragments = [
			'.comment-list'                       => $comment_html,
			'.azo-shop-reviews-pagination' => $pagination_html,
		];

		$response              = [];
		$response['fragments'] = apply_filters( 'azo/shop_reviews/fragments', $fragments );

		echo json_encode( $response );

		wp_die();
	}

	/**
	 * @param int    $offset
	 * @param int    $limit
	 * @param string $search_term
	 * @param string $order
	 *
	 * @return array|null|object
	 *
	 * get course reviews
	 *
	 * @since v.1.0.0
	 */
	public function get_shop_reviews( $offset = 0, $limit = 150, $search_term = '', $order = 'recent' ) {
		global $wpdb;

		switch ( $order ) {
			case 'lowest-rating':
				$sql_order_by = "ORDER BY rating ASC, comment_date DESC";
				break;
			case 'highest-rating';
				$sql_order_by = "ORDER BY rating DESC, comment_date DESC";
				break;
			default:
				$sql_order_by = "ORDER BY comment_date DESC";
				break;
		}

		$sql_content_like = '';
		if ( ! empty( $search_term ) ) {
			$sql_content_like = " AND comment.comment_content LIKE '%" . $wpdb->esc_like( $search_term ) . "%' ";
		}

		$sql_query = $wpdb->prepare( "SELECT comment.comment_ID, 
			comment.comment_post_ID, 
			comment.comment_author, 
			comment.comment_author_email, 
			comment.comment_date, 
			comment.comment_content, 
			comment.user_id, 
			comment.comment_approved, 
			comment.comment_parent, 
			review_rating.meta_value as rating,
			{$wpdb->users}.display_name 
			FROM {$wpdb->comments} as comment
			INNER JOIN {$wpdb->commentmeta} as review_rating
			ON comment.comment_ID = review_rating.comment_id 
			INNER JOIN {$wpdb->commentmeta} as verified
			ON comment.comment_ID = verified.comment_id
			LEFT JOIN {$wpdb->users}
			ON comment.user_id = {$wpdb->users}.ID 
			WHERE comment.comment_type = 'review' AND comment.comment_approved = '1'
			AND ( review_rating.meta_key = 'rating' AND ( verified.meta_key = 'verified' AND verified.meta_value = '1' )) 
			{$sql_content_like}
			{$sql_order_by}
			LIMIT {$offset},{$limit};" );

		$reviews = $wpdb->get_results( $sql_query );

		return $reviews;
	}

	/**
	 * @param string $search_term
	 *
	 * @return int
	 */
	public function get_shop_reviews_count( $search_term = '' ) {
		global $wpdb;

		$sql_content_like = '';
		if ( ! empty( $search_term ) ) {
			$sql_content_like = " AND {$wpdb->comments}.comment_content LIKE '%" . $wpdb->esc_like( $search_term ) . "%' ";
		}

		// Count.
		$count = $wpdb->get_var( $wpdb->prepare( "SELECT COUNT({$wpdb->comments}.comment_ID)
			FROM {$wpdb->comments}
			INNER JOIN {$wpdb->commentmeta} as rating
			ON {$wpdb->comments}.comment_ID = rating.comment_id 
			INNER JOIN {$wpdb->commentmeta} as verified
			ON {$wpdb->comments}.comment_ID = verified.comment_id
			LEFT JOIN {$wpdb->users}
			ON {$wpdb->comments}.user_id = {$wpdb->users}.ID 
			WHERE {$wpdb->comments}.comment_type = 'review' AND {$wpdb->comments}.comment_approved = '1'
			{$sql_content_like}
			AND ( rating.meta_key = 'rating' AND ( verified.meta_key = 'verified' AND verified.meta_value = '1' ));"
		) );

		return (int) $count;
	}

	public function get_shop_rating() {
		global $wpdb;

		$ratings = array(
			'rating_count'   => 0,
			'rating_sum'     => 0,
			'rating_avg'     => 0.00,
			'count_by_value' => array( 5 => 0, 4 => 0, 3 => 0, 2 => 0, 1 => 0 ),
		);

		$rating = $wpdb->get_row( $wpdb->prepare(
			"SELECT COUNT(rating.meta_value) AS rating_count,
					SUM(rating.meta_value) AS rating_sum
			FROM	{$wpdb->comments}
					INNER JOIN {$wpdb->commentmeta} AS rating
							ON {$wpdb->comments}.comment_ID = rating.comment_id
					INNER JOIN {$wpdb->commentmeta} AS verified
							ON {$wpdb->comments}.comment_ID = verified.comment_id
			WHERE 	{$wpdb->comments}.comment_approved = %s
					AND {$wpdb->comments}.comment_type = %s
					AND rating.meta_key = %s
					AND ( verified.meta_key = %s AND verified.meta_value = %s );
			",
			'1',
			'review',
			'rating',
			'verified',
			'1'
		) );

		if ( $rating->rating_count ) {
			$avg_rating = number_format( ( $rating->rating_sum / $rating->rating_count ), 2 );

			$stars = $wpdb->get_results( $wpdb->prepare(
				"SELECT commentmeta.meta_value AS rating,
						COUNT(commentmeta.meta_value) AS rating_count
				FROM	{$wpdb->comments} comments
						INNER JOIN {$wpdb->commentmeta} AS commentmeta
								ON comments.comment_ID = commentmeta.comment_id
						INNER JOIN {$wpdb->commentmeta} AS verified
								ON comments.comment_ID = verified.comment_id
				WHERE	1 =1 
						AND comments.comment_type = %s
						AND commentmeta.meta_key = %s
						AND ( verified.meta_key = %s AND verified.meta_value = %s )
				GROUP BY commentmeta.meta_value;
				",
				'review',
				'rating',
				'verified',
				'1'
			) );

			$ratings = array( 5 => 0, 4 => 0, 3 => 0, 2 => 0, 1 => 0 );

			foreach ( $stars as $star ) {
				$index = (int) $star->rating;

				array_key_exists( $index, $ratings ) ? $ratings[ $index ] = $ratings[ $index ] + $star->rating_count : 0;
			}

			$ratings = array(
				'rating_count'   => $rating->rating_count,
				'rating_sum'     => $rating->rating_sum,
				'rating_avg'     => $avg_rating,
				'count_by_value' => $ratings,
			);
		}

		return (object) $ratings;
	}

	public function get_shop_reviews_per_page() {
		return apply_filters( 'azo/shop_reviews/number_per_page', 5 );
	}

	public function output_shop_reviews_html() {
		wc_get_template( 'shop-reviews.php' );
	}

	public static function output_shop_review_html( $comment, $args, $depth ) {
		// phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
		$GLOBALS['comment'] = $comment;
		wc_get_template(
			'shop-review.php',
			array(
				'comment' => $comment,
				'args'    => $args,
				'depth'   => $depth,
			)
		);
	}
}

Shop_Reviews::instance()->initialize();
