<?php
/**
 * Azo Product Question
 *
 * @package  azo
 */

defined( 'ABSPATH' ) || exit;

/**
 * Product Question
 */
class Product_Question {
	protected static $instance = null;

	public static function instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	public function initialize() {
		// Support avatars for `question` comment type.
		add_filter( 'get_avatar_comment_types', array( $this, 'add_avatar_for_question_comment_type' ) );

		add_action( 'wp_ajax_nopriv_azo_add_comment', array( $this, 'add_comment' ) );
		add_action( 'wp_ajax_azo_add_comment', array( $this, 'add_comment' ) );

		add_action( 'wp_ajax_nopriv_azo_get_questions', array( $this, 'get_product_question_via_ajax' ) );
		add_action( 'wp_ajax_azo_get_questions', array( $this, 'get_product_question_via_ajax' ) );
	}

	/**
	 * Make sure WP displays avatars for comments with the `question` type.
	 *
	 * @since  2.3
	 * @param  array $comment_types Comment types.
	 * @return array
	 */
	public function add_avatar_for_question_comment_type( $comment_types ) {
		return array_merge( $comment_types, array( 'question' ) );
	}

	public function get_comment_per_page() {
		return apply_filters( 'azo/product_questions/per_page', get_option( 'comments_per_page' ), $this );
	}

	public function get_current_user() {
		global $user_ID;
		if ( $user_ID ) {
			$user = get_userdata( $user_ID );
		} else {
			$user = wp_set_current_user( 0 );
		}
		return $user;
	}

	public function get_user_current_display_name( $current_user ) {
		$display_name = trim( $current_user->display_name );
		if ( ! $display_name ) {
			$user_nicename = trim( $current_user->user_nicename );
			$display_name  = $user_nicename ? $user_nicename : trim( $current_user->user_login );
		}

		return $display_name;
	}

	public function get_comment_list_args( $args = array() ) {
		$args = wp_parse_args(
			$args,
			array(
				'style'       => 'ol',
				'callback'    => array( $this, 'question_template' ),
				'avatar_size' => 60,
				'echo'        => false,
			)
		);

		return $args;
	}

	public function add_comment() {
		$post_id            = isset( $_POST['post_id'] ) && $_POST['post_id'] ? absint( $_POST['post_id'] ) : 0;
		$question_parent_id = isset( $_POST['question_parent_id'] ) && $_POST['question_parent_id'] ? absint( $_POST['question_parent_id'] ) : 0;
		$question           = trim( $_POST['question'] );

		if ( $question ) {
			$current_user = $this->get_current_user();
			$website_url  = '';
			if ( $current_user && $current_user->ID ) {
				$user_id = $current_user->ID;
				$name    = $this->get_user_current_display_name( $current_user );
				$email   = $current_user->user_email;
			} else {
				$user_id     = 0;
				$name        = trim( $_POST['author'] );
				$email       = $_POST['email'];
				$website_url = ! empty( $_POST['url'] ) ? $_POST['url'] : '';
			}

			if ( $name && $email ) {
				//$user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : '';

				$data = array(
					'user_id'              => $user_id,
					'comment_post_ID'      => $post_id,
					'comment_author'       => $name,
					'comment_author_email' => $email,
					'comment_content'      => $question,
					'comment_author_url'   => $website_url,
					'comment_agent'        => $user_agent,
					'comment_type'         => 'question',
				);

				// Set type of reply comment ( children comments ) is comment instead of question.
				if ( $question_parent_id ) {
					$parent_comment         = get_comment( $question_parent_id );
					$parent_comment_id      = isset( $parent_comment->comment_ID ) ? $parent_comment->comment_ID : 0;
					$data['comment_parent'] = $parent_comment_id;
					$data['comment_type']   = 'comment';
				}

				$new_comment_id = wp_new_comment( wp_slash( $data ) );
				$new_comment    = get_comment( $new_comment_id );

				$list_comment_args = $this->get_comment_list_args();

				$response = array();

				$response['single_text'] = esc_html__( 'Question', 'azo' );
				$response['plural_text'] = esc_html__( 'Questions', 'azo' );

				$response['response'] = wp_list_comments( $list_comment_args, array( $new_comment ) );

				wp_send_json_success( $response );
			} else {
				wp_send_json_error( esc_html__( 'Some of field value is invalid', 'azo' ) );
			}
		} else {
			wp_send_json_error( esc_html__( 'Please fill out required fields.', 'azo' ) );
		}
	}

	public function get_product_question_via_ajax() {
		$keyword      =  ! empty( $_GET['keyword'] ) ? \Azo_Helper::data_clean( $_GET['keyword'] ) : '';
		$current_page = ! empty( $_GET['current_page'] ) ? intval( $_GET['current_page'] ) : 1;
		$post_id      = isset( $_GET['post_id'] ) && $_GET['post_id'] ? absint( $_GET['post_id'] ) : 0;

		$comments = get_comments(
			array(
				'type'    => 'question',
				'post_id' => $post_id,
				'status'  => 'approve',
			)
		);

		$new_comments = array();

		if ( $keyword ) {
			foreach ( $comments as $comment ) {
				if ( stripos( $comment->comment_content, $keyword ) !== false ) {
					// Retrieve questions which contain $keyword.
					$new_comments[] = $comment;
				}
			}
		} else {
			$new_comments = $comments;
		}

		$comments_children = array();

		foreach ( $new_comments as $new_comment ) {
			$_comments = get_comments(
				array(
					'parent'       => $new_comment->comment_ID,
					'hierarchical' => true,
					'status'       => 'approve',
				)
			);

			foreach ( $_comments as $_comment ) {
				$comments_children[] = $_comment;
			}
		}

		$all_comments = array_merge( $new_comments, $comments_children );

		$comment_count = count( $new_comments );

		$list_comment_args = $this->get_comment_list_args();

		// Comment Pagination.
		$per_page    = $this->get_comment_per_page();
		$total_pages = ceil( $comment_count / $per_page );

		if ( $current_page > $total_pages ) {
			$current_page = $total_pages;
		}

		$list_comment_args['page']     = $current_page;
		$list_comment_args['per_page'] = $per_page;

		ob_start();
		if ( get_comment_pages_count( $new_comments ) > 1 && get_option( 'page_comments' ) ) {
			?>
			<nav class="navigation question-navigation comment-navigation">
				<h2 class="screen-reader-text"><?php esc_html_e( 'Question navigation', 'azo' ); ?></h2>

				<div class="comment-nav-links">
					<?php
					\Azo_Templates::render_paginate_links(
						array(
							'format'  => '?current_page=%#%',
							'current' => max( 1, $current_page ),
							'total'   => $total_pages,
						)
					);
					?>
				</div>
			</nav>
			<?php
		}

		$pagination_html = ob_get_clean();

		$response = array();

		if ( 1 === $comment_count ) {
			$response['text'] = esc_html__( 'Question', 'azo' );
		} elseif ( $comment_count > 1 ) {
			$response['text'] = esc_html__( 'Questions', 'azo' );
		}

		$response['count'] = $comment_count;

		$response['response']   = wp_list_comments( $list_comment_args, $all_comments );
		$response['pagination'] = $pagination_html;
		$response['no_found']   = esc_html__( 'No questions found.', 'azo' );

		wp_send_json( $response );
		wp_die();
	}

	/**
	 * Question Callback
	 *
	 * @param  [type] $comment type.
	 * @param  [type] $args    type.
	 * @param  [type] $depth   type.
	 */
	public function question_template( $comment, $args, $depth ) {
		$GLOBALS['comment'] = $comment;
		?>

		<li <?php echo esc_attr( comment_class() ); ?> id="li-comment-<?php echo esc_attr( comment_ID() ); ?>">
		<div id="comment-<?php echo esc_attr( comment_ID() ); ?>" class="comment-wrap">
			<div class="woo-comment-author">
				<?php echo get_avatar( $comment, $args['avatar_size'] ); ?>
				<div class="meta">
					<?php printf( '<h6 class="fn">%s</h6>', get_comment_author_link() );?>
					<time class="comment-datetime">
						<?php printf( esc_html__( '%s at %s', 'azo' ), get_comment_date(), get_comment_time() ); ?>
					</time>
				</div>
			</div>
			<div class="woo-comment-content">
				<?php if ( '0' === $comment->comment_approved ) : ?>
					<em class="comment-awaiting-messages"><?php esc_html_e( 'Your question is awaiting moderation.', 'azo' ); ?></em>
					<br/>
				<?php endif; ?>
				<div class="comment-text"><?php comment_text(); ?></div>
			</div>
			<div class="comment-footer woo-comment-footer">
				<?php
				$this->question_reply_and_cancel_link(
					array(
						'depth'     => $depth,
						'max_depth' => $args['max_depth'],
					),
					$comment
				);
				?>
			</div>
		</div>
		<?php
	}

	/**
	 * Question reply and cancel link
	 *
	 * @param  array  $args    type.
	 * @param  [type] $comment type.
	 */
	function question_reply_and_cancel_link( $args = array(), $comment ) {
		$defaults = array(
			'max_depth' => 0,
			'depth'     => 0,
		);

		$args = wp_parse_args( $args, $defaults );

		if ( 0 == $args['depth'] || $args['max_depth'] <= $args['depth'] ) {
			return;
		}

		$enable = apply_filters( 'azo/enable_question_reply_link', true );
		if ( ! $enable ) {
			return;
		}

		$comment = get_comment( $comment );

		if ( empty( $comment ) ) {
			return;
		}

		if ( empty( $post ) ) {
			$post = $comment->comment_post_ID;
		}

		$post = get_post( $post );

		if ( ! comments_open( $post->ID ) ) {
			return;
		}

		echo '<div class="reply-action">';

		echo sprintf(
			'<a href="%s" class="question-reply question-reply--%s" data-post-id="%s" data-question-id="%s">%s</a>',
			add_query_arg(
				array(
					'reply_to_question' => $comment->comment_ID,
				),
				get_permalink( $comment->comment_post_ID )
			),
			esc_attr( $comment->comment_ID ),
			esc_attr( $comment->comment_post_ID ),
			esc_attr( $comment->comment_ID ),
			esc_html__( 'Reply', 'azo' )
		);

		echo sprintf(
			'<a href="#" class="cancel-reply cancel-reply--%s" data-question-id="%s">%s</a>',
			esc_attr( $comment->comment_ID ),
			esc_attr( $comment->comment_ID ),
			esc_html__( 'Cancel Reply', 'azo' )
		);

		echo '</div>';
	}

	/**
	 * Question Form
	 */
	public function question_form() {
		$req      = get_option( 'require_name_email' );
		$aria_req = '';
		$required = '';
		if ( $req ) {
			$aria_req = " aria-required='true'";
			$required = '<span class="required">*</span>';
		}

		$reply_to_id = isset( $_GET['reply_to_question'] ) ? (int) $_GET['reply_to_question'] : 0;

		?>
		<form method="post" id="question-form" class="question-form" action="">

			<?php if ( ! is_user_logged_in() ) : ?>
			<div class="row">
				<div class="question-form-author col-sm-6">
					<label for="q-author"><?php echo esc_html__( 'Name', 'azo' ) . $required; ?></label>
					<input type="text" name="q-author" value="" id="q-author" <?php echo '' . $aria_req ?>>
				</div>
				<div class="question-form-email col-sm-6">
					<label for="q-email"><?php echo esc_html__( 'Email', 'azo' ) . $required; ?></label>
					<input type="email" name="q-email" value="" id="q-email" <?php echo '' . $aria_req ?>>
				</div>
			</div>
			<?php endif; ?>

			<div class="question-form-question">
				<label for="question"><?php echo esc_html__( 'Question', 'azo' ) . $required; ?></label>
				<textarea id="question" name="question" cols="45" rows="8" aria-required="true"></textarea>
			</div>
			<div class="question-form-submit form-submit">
				<input type="submit" name="submit" value="<?php echo esc_attr__( 'Submit', 'azo' ); ?>" class="submit">
				<input type="hidden" name="post_id" value="<?php echo get_the_ID(); ?>">
				<input type="hidden" name="question_parent_id" value="<?php echo esc_attr( $reply_to_id ); ?>">
			</div>
		</form>
		<?php
	}

	/**
	 * Render Question List
	 */
	public function question_list() {
		$comments = get_comments(
			array(
				'type'    => 'question',
				'post_id' => get_the_ID(),
				'status'  => 'approve',
			)
		);

		$comments_children = array();

		foreach ( $comments as $comment ) {
			$_comments = get_comments(
				array(
					'parent'       => $comment->comment_ID,
					'hierarchical' => true,
					'status'       => 'approve',
				)
			);

			foreach ( $_comments as $_comment ) {
				$comments_children[] = $_comment;
			}
		}

		$all_comments = array_merge( $comments, $comments_children );

		// Parent Comment Count.
		$comment_count = count( $comments );

		// Comment list args.
		$comment_list_args = $this->get_comment_list_args();

		$container_class = 'question-list-container';

		if ( ! $comment_count ) {
			$container_class .= ' no-question';
		}

		global $product;
		?>
		<div class="<?php echo esc_attr( $container_class ); ?>">
			<?php reviews_summary( $product ); ?>

			<div class="question-form-wrapper">
				<?php $this->question_form(); ?>
				<div class="question-form-message-box"></div>
			</div>

			<div class="question-toolbar">
				<h4 class="question-title">
					<span class="question-count"><?php echo intval( $comment_count ); ?></span> 
					<span class="question-text"><?php echo _n( 'Question', 'Questions', $comment_count, 'azo' ); ?></span>
				</h4>

				<?php if ( $comment_count > 1 ) : ?>
				<div class="question-filter">
					<form class="question-search-form" method="get" action="">
						<label for="question-search-form__filter-content" class="hidden"><?php esc_html_e( 'Search', 'azo' ); ?></label>
						<input 
						id="question-search-form__filter-content" 
						type="text" 
						name="keyword" 
						placeholder="<?php echo esc_attr__( 'Search', 'azo' ); ?>">
						<input type="hidden" name="action" value="azo_get_questions">
						<input type="hidden" name="current_page" value="1">
						<input type="hidden" name="post_id" value="<?php echo get_the_ID(); ?>">
					</form>
				</div>
				<?php endif; ?>
			</div>

			<div class="question-list-wrapper">
				<ol id="question-list" class="comment-list question-list">
					<?php
					if ( $comment_count ) {
						echo wp_list_comments( $comment_list_args, $all_comments );
					} else {
						echo '<li class="woocommerce-noreviews"><p>' . esc_html__( 'There are no question yet.', 'azo' ) . '</p></li>';
					}
					?>
				</ol>
				<?php
				if ( get_comment_pages_count( $comments ) > 1 && get_option( 'page_comments' ) ) {
					$current_page = 1;
					$per_page     = $this->get_comment_per_page();
					$total_pages  = ceil( $comment_count / $per_page );
					?>
					<nav class="navigation question-navigation comment-navigation">
						<h2 class="screen-reader-text"><?php esc_html_e( 'Question navigation', 'azo' ); ?></h2>

						<div class="comment-nav-links">
							<?php
							\Azo_Templates::render_paginate_links(
								array(
									'format'  => '?current_page=%#%',
									'current' => max( 1, $current_page ),
									'total'   => $total_pages,
								)
							);
							?>
						</div>
					</nav>
					<?php
				}
				?>
			</div>
		</div>
		<?php
	}
}
Product_Question::instance()->initialize();
