<?php
/**
 * Azo Discount
 *
 * @package  azo
 */

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'Discount' ) ) {
	/**
	 * Discount
	 */
	class Discount {

		/**
		 * Instance Variable
		 *
		 * @var instance
		 */
		protected static $instance = null;

		/**
		 *  Initiator
		 */
		public static function instance() {
			if ( null === self::$instance ) {
				self::$instance = new self();
			}

			return self::$instance;
		}

		/**
		 * Initialize
		 */
		public function initialize() {
			if ( ! $this->is_activate() ) {
				return;
			}

			/**
			 * Remove quantity based pricing table from plugin then add custom HTML
			 *
			 * @see WAD_Discount::get_quantity_pricing_tables()
			 * @see add_quantity_pricing_tables_html()
			 */
			add_filter( 'wad_get_quantity_pricing_tables', '__return_empty_string', 99, 3 );
			add_action( 'woocommerce_before_add_to_cart_button', [ $this, 'add_quantity_pricing_tables_html' ] );
		}

		/**
		 * Activate
		 */
		public function is_activate() {
			if ( defined( 'WAD_VERSION' ) ) {
				return true;
			}

			return false;
		}

		/**
		 * Add quantity pricing tables html
		 */
		public function add_quantity_pricing_tables_html() {
			global $product;
			$product_id       = $product->get_id();
			$quantity_pricing = get_post_meta( $product_id, 'o-discount', true );
			$rules_type       = get_proper_value( $quantity_pricing, 'rules-type', 'intervals' );

			if ( ! isset( $quantity_pricing['enable'] ) ) {
				return;
			}

			if ( ! isset( $quantity_pricing['rules'] ) && ! isset( $quantity_pricing['rules-by-step'] ) ) {
				return;
			}
			?>
			<?php
			if ( 'intervals' === $rules_type ) {
				if ( $product->get_type() === 'variable' ) {
					$available_variations = $product->get_available_variations();
					foreach ( $available_variations as $variation ) {
						$product_price = $variation['display_price'];
						$this->get_quantity_pricing_table( $variation['variation_id'], $quantity_pricing, $product_price );
					}
				} else {
					$product_price = $product->get_price();
					$this->get_quantity_pricing_table( $product_id, $quantity_pricing, $product_price, true );
				}
			} elseif ( 'steps' === $rules_type ) {

				if ( $product->get_type() === 'variable' ) {
					$available_variations = $product->get_available_variations();
					foreach ( $available_variations as $variation ) {
						$product_price = $variation['display_price'];
						$this->get_steps_quantity_pricing_table( $variation['variation_id'], $quantity_pricing, $product_price );
					}
				} else {
					$product_price = $product->get_price();
					$this->get_steps_quantity_pricing_table( $product_id, $quantity_pricing, $product_price, true );
				}
			}
		}

		/**
		 * Steps quantity pricing table
		 *
		 * @param  [type]  $product_id       type.
		 * @param  [type]  $quantity_pricing type.
		 * @param  [type]  $product_price    type.
		 * @param  boolean $display          type.
		 */
		private function get_steps_quantity_pricing_table( $product_id, $quantity_pricing, $product_price, $display = false ) {
			global $product;
			$classes = 'quantity-discount-table wad-qty-pricing-table';
			?>
			<table class="<?php echo esc_attr( $classes ); ?>" data-id="<?php echo esc_attr( $product_id ); ?>"
				<?php if ( ! $display ) : ?>
					style="display: none;"
				<?php endif; ?>
			>
				<thead>
				<tr>
					<th class="col-quantity-discount-name">&nbsp;</th>
					<th class="col-quantity-regular-price">&nbsp;</th>
					<th class="col-quantity-discount-price">&nbsp;</th>
					<th class="col-quantity-discount-badge">&nbsp;</th>
				</tr>
				</thead>
				<tbody>
				<?php
				$discount_price = $product_price;
				$discount_type  = $quantity_pricing['type'];
				foreach ( $quantity_pricing['rules-by-step'] as $rule ) :
					if ( '' === $rule['every'] ) {
						continue;
					}

					$input_disable = false;
					$min_quantity  = apply_filters( 'woocommerce_quantity_input_min', $product->get_min_purchase_quantity(), $product );
					$max_quantity  = apply_filters( 'woocommerce_quantity_input_max', $product->get_max_purchase_quantity(), $product );
					$quantity      = intval( $rule['every'] );

					if ( $quantity < $min_quantity || ( $quantity > $max_quantity && -1 !== $max_quantity ) ) {
						$input_disable = true;
					}

					$discount_value_text = $rule['discount'];

					switch ( $discount_type ) {
						case 'fixed':
							$discount_price = $product_price - $rule['discount'];

							$discount_percentage = round( ( $product_price - $discount_price ) / $product_price * 100 );

							$discount_value_text = $discount_percentage . '%';
							break;
						case 'percentage':
							$discount_price = $product_price - ( $product_price * $rule['discount'] ) / 100;

							$discount_value_text = $rule['discount'] . '%';
							break;
					}

					$quantity_regular_price = $product_price * $quantity;
					$quantity_sale_price    = $discount_price * $quantity;

					$row_classes = 'quantity-discount-item';

					if ( $input_disable ) {
						$row_classes .= ' disabled';
					}
					?>
					<tr class="<?php echo esc_attr( $row_classes ); ?>">
						<td class="col-quantity-discount-name">
							<label class="quantity-discount-name">
								<input type="radio" name="quantity_discount" value="<?php echo esc_attr( $quantity ); ?>"
								<?php
								if ( $input_disable ) {
									echo 'disabled';
								}
								?>
								/>
								<?php echo sprintf( esc_html__( 'Buy %s :', 'azo' ), $quantity ); ?>
							</label>
						</td>
						<td class="col-quantity-regular-price">
							<div class="price quantity-regular-price">
								<del>
									<?php echo wc_price( $quantity_regular_price ); ?>
								</del>
							</div>
						</td>
						<td class="col-quantity-discount-price">
							<div class="price quantity-discount-price">
								<ins>
									<?php echo wc_price( $quantity_sale_price ); ?>
								</ins>
							</div>
						</td>
						<td class="col-quantity-discount-badge">
							<div class="quantity-discount-badge">
								<?php echo sprintf( esc_html__( 'Save %s', 'azo' ), $discount_value_text ); ?>
							</div>
						</td>
					</tr>
				<?php endforeach; ?>
				</tbody>
			</table>
			<?php
		}

		private function get_quantity_pricing_table( $product_id, $quantity_pricing, $product_price, $display = false ) {
			global $product;
			$classes = 'quantity-discount-table wad-qty-pricing-table';
			?>
			<table class="<?php echo esc_attr( $classes ); ?>" data-id="<?php echo esc_attr( $product_id ); ?>"
				<?php if ( ! $display ) : ?>
					style="display: none;"
				<?php endif; ?>
			>
				<thead>
				<tr>
					<th class="col-quantity-discount-name">&nbsp;</th>
					<th class="col-quantity-regular-price">&nbsp;</th>
					<th class="col-quantity-discount-price">&nbsp;</th>
					<th class="col-quantity-discount-badge">&nbsp;</th>
				</tr>
				</thead>
				<tbody>
				<?php
				$discount_price = $product_price;
				$discount_type  = $quantity_pricing['type'];

				foreach ( $quantity_pricing['rules'] as $rule ) :
					$discount_label = '';
					$min_quantity   = apply_filters( 'woocommerce_quantity_input_min', $product->get_min_purchase_quantity(), $product );
					$max_quantity   = apply_filters( 'woocommerce_quantity_input_max', $product->get_max_purchase_quantity(), $product );
					$input_disable  = false;

					if ( '' !== $rule['min'] ) {
						$quantity = $rule['min'];
					} elseif ( '' !== $rule['max'] ) {
						$quantity = $rule['max'];
					}

					if ( empty( $quantity ) ) {
						continue;
					}

					if ( $quantity < $min_quantity || ( $quantity > $max_quantity && -1 !== $max_quantity ) ) {
						$input_disable = true;
					}

					if ( '' !== $rule['min'] && '' !== $rule['max'] ) {
						if ( $rule['min'] !== $rule['max'] ) {
							$discount_label = sprintf( esc_html__( 'Buy %1$s to %2$s:', 'azo' ), $rule['min'], $rule['max'] );
						} else {
							$discount_label = sprintf( esc_html__( 'Buy %s :', 'azo' ), $quantity );
						}
					} elseif ( '' !== $rule['min'] ) {
						$discount_label = sprintf( esc_html__( 'Buy from %s :', 'azo' ), $quantity );
					} elseif ( '' !== $rule['max'] ) {
						$discount_label = sprintf( esc_html__( 'Buy %s :', 'azo' ), $quantity );
					}

					$discount_value_text = $rule['discount'];

					switch ( $discount_type ) {
						case 'fixed':
							$discount_price = $product_price - $rule['discount'];

							$discount_percentage = round( ( $product_price - $discount_price ) / $product_price * 100 );
							$discount_value_text = $discount_percentage . '%';
							break;
						case 'percentage':
							$discount_price = $product_price - ( $product_price * $rule['discount'] ) / 100;

							$discount_value_text = $rule['discount'] . '%';
							break;
					}

					$quantity               = intval( $quantity );
					$quantity_regular_price = $product_price * $quantity;
					$quantity_sale_price    = $discount_price * $quantity;

					$row_classes = 'quantity-discount-item';

					if ( $input_disable ) {
						$row_classes .= ' disabled';
					}
					?>
					<tr class="<?php echo esc_attr( $row_classes ); ?>">
						<td class="col-quantity-discount-name">
							<label class="quantity-discount-name">
								<input type="radio" name="quantity_discount" value="<?php echo esc_attr( $quantity ); ?>" 
								<?php
								if ( $input_disable ) {
									echo 'disabled';
								}
								?>
								/>
								<?php echo esc_html( $discount_label ); ?>
							</label>
						</td>
						<td class="col-quantity-regular-price">
							<div class="price quantity-regular-price">
								<del><?php echo wc_price( $quantity_regular_price ); ?></del>
							</div>
						</td>
						<td class="col-quantity-discount-price">
							<div class="price quantity-discount-price">
								<ins><?php echo wc_price( $quantity_sale_price ); ?></ins>
							</div>
						</td>
						<td class="col-quantity-discount-badge">
							<div class="quantity-discount-badge">
								<?php echo sprintf( esc_html__( 'Save %s', 'azo' ), $discount_value_text ); ?>
							</div>
						</td>
					</tr>
				<?php endforeach; ?>
				</tbody>
			</table>
			<?php
		}
	}

	Discount::instance()->initialize();
}
