<?php
/**
 * Elementor Widget
 *
 * @package azo
 */

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'Azo_Elementor_Widgets' ) ) {
	/**
	 * Main Elementor Widgets Class
	 */
	class Azo_Elementor_Widgets {
		/**
		 * Instance
		 *
		 * @var instance
		 */
		private static $instance;

		/**
		 *  Initiator
		 */
		public static function init() {
			if ( ! isset( self::$instance ) ) {
				self::$instance = new self();
			}
			return self::$instance;
		}

		/**
		 * Woostify Pro Constructor.
		 */
		public function __construct() {
			add_action( 'wp_ajax_azo_autocomplete_selected', array( $this, 'azo_autocomplete_selected' ) );
			add_action( 'wp_ajax_azo_autocomplete_search', array( $this, 'azo_autocomplete_search' ) );
			add_action( 'elementor/controls/controls_registered', array( $this, 'add_custom_controls' ) );
			add_action( 'elementor/preview/enqueue_scripts', array( $this, 'elementor_preview_register_scripts' ) );
			add_action( 'elementor/frontend/after_register_scripts', array( $this, 'elementor_front_end_register_scripts' ) );
		}

		/**
		 * Add custom controls
		 */
		function add_custom_controls() {
			$controls_manager = \Elementor\Plugin::$instance->controls_manager;

			require_once AZO_DIR . 'inc/elementor-controls/class-azo-autocomplete-control.php';

			$controls_manager->register_control( 'autocomplete', new Azo_Autocomplete_Control() );
		}

		/**
		 * Selected
		 */
		function azo_autocomplete_selected() {
			check_ajax_referer( 'azo-autocomplete', 'security_nonce' );

			$query       = isset( $_POST['query'] ) ? sanitize_text_field( wp_unslash( $_POST['query'] ) ) : '';
			$value       = isset( $_POST['value'] ) ? sanitize_text_field( wp_unslash( $_POST['value'] ) ) : '';
			$selected_id = isset( $_POST['selected_id'] ) ? sanitize_text_field( wp_unslash( $_POST['selected_id'] ) ) : '';
			if ( ! in_array( $query, array( 'post_type', 'term' ), true ) || ! current_user_can( 'edit_theme_options' ) || ! $value || ! $selected_id ) {
				wp_send_json_error();
			}

			$selected_id = explode( ',', $selected_id );

			ob_start();
			foreach ( $selected_id as $k ) {
				$data_name = get_the_title( (int) $k );

				if ( 'term' === $query ) {
					$term      = get_term_by( 'id', (int) $k, $value );
					$data_name = $term->name;
				}
				?>
				<span class="wty-autocomplete-id" data-id="<?php echo esc_attr( $k ); ?>">
					<?php echo esc_html( $data_name ); ?>
					<i class="wty-autocomplete-remove-id eicon-close-circle"></i>
				</span>
				<?php
			}

			wp_send_json_success( ob_get_clean() );
		}

		/**
		 * Searching
		 */
		function azo_autocomplete_search() {
			check_ajax_referer( 'azo-autocomplete', 'security_nonce' );

			$query = isset( $_POST['query'] ) ? sanitize_text_field( wp_unslash( $_POST['query'] ) ) : '';
			$value = isset( $_POST['value'] ) ? sanitize_text_field( wp_unslash( $_POST['value'] ) ) : '';
			if ( ! in_array( $query, array( 'post_type', 'term' ), true ) || ! current_user_can( 'edit_theme_options' ) || ! $value ) {
				wp_send_json_error();
			}

			$keyword = isset( $_POST['keyword'] ) ? sanitize_text_field( wp_unslash( $_POST['keyword'] ) ) : '';

			ob_start();
			if ( 'term' === $query ) {
				$args = array(
					'hide_empty' => true,
					'search'     => $keyword,
				);
				$cats = get_terms( $value, $args );

				if ( ! empty( $cats ) && ! is_wp_error( $cats ) ) {
					foreach ( $cats as $k ) {
						?>
						<span class="wty-autocomplete-id" data-id="<?php echo esc_attr( $k->term_id ); ?>">
							<?php echo esc_html( $k->name ); ?>
						</span>
						<?php
					}
				} else {
					?>
					<span class="no-posts-found"><?php esc_html_e( 'Nothing Found', 'azo' ); ?></span>
					<?php
				}
			} else {
				$args = array(
					'post_type'           => $value,
					'post_status'         => 'publish',
					'ignore_sticky_posts' => 1,
					'posts_per_page'      => -1,
					's'                   => $keyword,
				);

				$products = new \WP_Query( $args );

				if ( $products->have_posts() ) {
					while ( $products->have_posts() ) {
						$products->the_post();
						?>
						<span class="wty-autocomplete-id" data-id="<?php the_ID(); ?>">
							<?php the_title(); ?>
						</span>
						<?php
					}

					wp_reset_postdata();
				} else {
					?>
					<span class="no-posts-found"><?php esc_html_e( 'Nothing Found', 'azo' ); ?></span>
					<?php
				}
			}

			$res = ob_get_clean();

			wp_send_json_success( $res );
		}

		/**
		 * Register preview mode scripts
		 */
		public function elementor_preview_register_scripts() {
			// Scripts.
			wp_enqueue_script( 'azo-elementor-widget' );
			wp_enqueue_script( 'azo-countdown' );

			// Styles.
			wp_enqueue_style( 'animate' );
		}

		/**
		 * Register elementor scripts
		 */
		public function elementor_front_end_register_scripts() {
			// Elementor widgets js.
			wp_register_script(
				'azo-elementor-widget',
				AZO_URI . 'modules/elementor/assets/js/elementor-widgets' . azo_suffix() . '.js',
				array( 'tiny-slider', 'azo-countdown' ),
				AZO_VERSION,
				true
			);

			// Animate animation.
			wp_register_style(
				'animate',
				AZO_URI . 'modules/elementor/assets/css/animate.css',
				array(),
				AZO_VERSION
			);

			// Elementor widgets.
			wp_register_style(
				'azo-elementor-widgets',
				AZO_URI . 'modules/elementor/assets/css/azo-elementor-widgets.css',
				array(),
				AZO_VERSION
			);
		}
	}

	Azo_Elementor_Widgets::init();
}
