<?php
/**
 * Helpers
 *
 * @package azo
 */

// @codingStandardsIgnoreStart

defined( 'ABSPATH' ) || exit;

// SET CONTENT WIDTH.
if ( ! isset( $content_width ) ) {
	$content_width = 1200;
}

if ( ! function_exists( 'azo_get_page_id' ) ) {
	/**
	 * Get page id
	 *
	 * @return int $page_id Page id
	 */
	function azo_get_page_id() {
		$page_id = get_queried_object_id();

		if ( class_exists( 'woocommerce' ) && is_shop() ) {
			$page_id = get_option( 'woocommerce_shop_page_id' );
		}

		return $page_id;
	}
}

if ( ! function_exists( 'azo_suffix' ) ) {
	/**
	 * Define Script debug.
	 *
	 * @return     string $suffix
	 */
	function azo_suffix() {
		$suffix = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? '' : '.min';

		return $suffix;
	}
}

/**
 * Loads a template part into a template with prefix folder given.
 *
 * @see get_template_part()
 *
 * @param string $slug The slug name for the generic template.
 * @param string $name The name of the specialised template.
 * @param array  $args Optional. Additional arguments passed to the template.
 *                     Default empty array.
 *
 * @return void|false Void on success, false if the template does not exist.
 */
function azo_load_template( $slug, $name = null, $args = array() ) {
	get_template_part( "template-parts/{$slug}", $name, $args );
}

if ( ! function_exists( 'azo_preloader' ) ) {
	/**
	 * PRELOADER
	 */
	function azo_preloader() {
		if ( true === get_theme_mod( 'loading', false ) ) {
			$image_preload = get_theme_mod( 'image_preload_setting_url', '' );

			// Loading text.
			$loading_text = get_theme_mod( 'loading_text', 'Boostify' );

			// Don't render template in Elementor editor mode.
			if ( class_exists( '\Elementor\Plugin' ) && \Elementor\Plugin::$instance->preview->is_preview_mode() ) {
				return;
			}

			$style = get_theme_mod( 'pre_loader_style' );

			if ( 'random' === $style ) {
				$style = array_rand( Azo_Helper::$preloader_style );
			}
			?>
			<div id="preloader">
				<div class="preloader-logo">
					<?php if ( '' !== $image_preload ) { ?>
						<img src="<?php echo esc_url( $image_preload ); ?>" alt="<?php esc_attr_e( 'logo-loader-azo', 'azo' ); ?>" class="logo-img-loader">
					<?php } ?>

					<!--preloader circle -->
					<div id="page-preloader" class="page-loading clearfix">
						<div class="page-load-inner">
							<div class="preloader-wrap">
								<div class="wrap-2">
									<div class="inner">
										<?php azo_load_template( 'preloader/style', $style ); ?>
									</div>
								</div>
							</div>
						</div>
					</div>

					<div class="azo-demo-loader">
						<?php echo esc_html( $loading_text ); ?>
					</div>
				</div>
			</div>
			<?php
		}
	}
}


if ( ! function_exists( 'azo_label' ) ) {
	/**
	 * SET KIRKI LABEL
	 *
	 * @param string $data type.
	 */
	function azo_label( $data = '' ) {
		$output = '<span style="padding: 15px 10px; background-color: #cbcbcb; color: #333; display: block; text-transform: uppercase; font-weight: 700; margin: 15px 0;">' . $data . '</span>';

		return $output;
	}
}

if ( ! function_exists( 'azo_img_alt' ) ) {
	/**
	 * SET IMAGE ALT
	 *
	 * @param  [type] $id  type.
	 * @param  string $alt type.
	 */
	function azo_img_alt( $id = null, $alt = '' ) {
		$data    = get_post_meta( $id, '_wp_attachment_image_alt', true );
		$img_alt = ! empty( $data ) ? $data : $alt;

		return $img_alt;
	}
}

if ( ! function_exists( 'azo_logo_image' ) ) {
	/**
	 * SET LOGO IMAGE
	 */
	function azo_logo_image() {
		$pid         = get_queried_object_id();
		$p_lg        = function_exists( 'fw_get_db_post_option' ) ? fw_get_db_post_option( $pid, 'p_lg' ) : '';
		$menu_layout = azo_menu_slug();

		/*general logo*/
		$logo = get_theme_mod( 'custom_logo' );

		/*logo src*/
		$logo_src_full = wp_get_attachment_image_src( $logo, 'full' );
		$logo_src      = ! empty( $logo ) ? $logo_src_full[0] : get_template_directory_uri() . '/images/logo.svg';

		// Retina logo.
		$retina_logo     = get_theme_mod( 'retina_logo' );
		$retina_logo_src = ! empty( $retina_logo ) ? $retina_logo : $logo_src;

		$tag       = 'figure';
		$child_tag = 'figcaption';

		if ( is_front_page() ) {
			$tag       = 'h1';
			$child_tag = 'span';
		}

		?>
		<<?php echo esc_attr( $tag ) . ' '; ?> class="theme-logo" itemscope itemtype="https://schema.org/Organization">
		<a class="logo" href="<?php echo esc_url( home_url( '/' ) ); ?>" itemprop="url">
			<img class="primary-logo" src="<?php echo esc_url( $logo_src ); ?>" alt="<?php esc_attr_e( 'Logo image', 'azo' ); ?>" itemprop="logo" srcset="<?php echo esc_url( $logo_src ); ?> 1x, <?php echo esc_url( $retina_logo_src ); ?> 2x" >
		</a>
		<<?php echo esc_attr( $child_tag ); ?> class="screen-reader-text"><?php echo esc_html( bloginfo( 'name' ) ); ?></<?php echo esc_html( $child_tag ); ?>>
		</<?php echo esc_attr( $tag ); ?>>
		<?php
	}
}

if ( ! function_exists( 'azo_schema_markup' ) ) {
	/**
	 * SCHEMA MARKUP SHORTCUT
	 *
	 * @param  [type] $type type.
	 */
	function azo_schema_markup( $type ) {

		if ( empty( $type ) ) {
			return;
		}

		$attributes = '';
		$attr       = array();

		switch ( $type ) {
			case 'head':
				$attr['itemscope'] = 'itemscope';
				$attr['itemtype']  = 'https://schema.org/WebSite';
				break;

			case 'body':
				$attr['itemscope'] = 'itemscope';
				$attr['itemtype']  = 'https://schema.org/WebPage';
				break;

			case 'header':
				$attr['role']      = 'banner';
				$attr['itemscope'] = 'itemscope';
				$attr['itemtype']  = 'https://schema.org/WPHeader';
				break;

			case 'nav':
				$attr['role']      = 'navigation';
				$attr['itemscope'] = 'itemscope';
				$attr['itemtype']  = 'https://schema.org/SiteNavigationElement';
				break;

			case 'breadcrumb':
				$attr['itemscope'] = 'itemscope';
				$attr['itemtype']  = 'https://schema.org/BreadcrumbList';
				break;

			case 'title':
				$attr['itemprop'] = 'headline';
				break;

			case 'sidebar':
				$attr['role']      = 'complementary';
				$attr['itemscope'] = 'itemscope';
				$attr['itemtype']  = 'https://schema.org/WPSideBar';
				break;

			case 'footer':
				$attr['role']      = 'contentinfo';
				$attr['itemscope'] = 'itemscope';
				$attr['itemtype']  = 'https://schema.org/WPFooter';
				break;

			case 'main':
				$attr['itemprop'] = 'mainContentOfPage';
				if ( is_search() ) {
					$attr['itemtype'] = 'https://schema.org/SearchResultsPage';
				}
				break;

			case 'article':
				$attr['itemscope'] = 'itemscope';
				$attr['itemtype']  = 'https://schema.org/Article';
				break;

			case 'blog':
				$attr['itemscope'] = 'itemscope';
				$attr['itemtype']  = 'https://schema.org/Blog';
				break;

			case 'blog_list':
				$attr['itemscope'] = 'itemscope';
				$attr['itemtype']  = 'https://schema.org/BlogPosting';
				break;

			case 'creative_work':
				$attr['itemscope'] = 'itemscope';
				$attr['itemtype']  = 'https://schema.org/CreativeWork';
				break;

			case 'author':
				$attr['itemprop']  = 'author';
				$attr['itemscope'] = 'itemscope';
				$attr['itemtype']  = 'https://schema.org/Person';
				break;

			case 'person':
				$attr['itemscope'] = 'itemscope';
				$attr['itemtype']  = 'https://schema.org/Person';
				break;

			case 'comment':
				$attr['itemprop']  = 'comment';
				$attr['itemscope'] = 'itemscope';
				$attr['itemtype']  = 'https://schema.org/UserComments';
				break;

			case 'comment_author':
				$attr['itemprop']  = 'creator';
				$attr['itemscope'] = 'itemscope';
				$attr['itemtype']  = 'https://schema.org/Person';
				break;

			case 'comment_author_link':
				$attr['itemprop']  = 'creator';
				$attr['itemscope'] = 'itemscope';
				$attr['itemtype']  = 'https://schema.org/Person';
				$attr['rel']       = 'external nofollow';
				break;

			case 'comment_time':
				$attr['itemprop']  = 'commentTime';
				$attr['itemscope'] = 'itemscope';
				$attr['datetime']  = get_the_time( 'c' );
				break;

			case 'comment_text':
				$attr['itemprop'] = 'commentText';
				break;

			case 'author_box':
				$attr['itemprop']  = 'author';
				$attr['itemscope'] = 'itemscope';
				$attr['itemtype']  = 'https://schema.org/Person';
				break;

			case 'video':
				$attr['itemprop'] = 'video';
				$attr['itemtype'] = 'https://schema.org/VideoObject';
				break;

			case 'audio':
				$attr['itemscope'] = 'itemscope';
				$attr['itemtype']  = 'https://schema.org/AudioObject';
				break;

			case 'image':
				$attr['itemscope'] = 'itemscope';
				$attr['itemprop']  = 'image';
				$attr['itemtype']  = 'https://schema.org/ImageObject';
				break;

			case 'organization':
				$attr['itemprop']  = 'organization';
				$attr['itemscope'] = 'itemscope';
				$attr['itemtype']  = 'https://schema.org/Organization';
				break;

			case 'publisher':
				$attr['itemprop']  = 'publisher';
				$attr['itemscope'] = 'itemscope';
				$attr['itemtype']  = 'https://schema.org/Organization';
				break;

			case 'logo':
				$attr['itemprop']  = 'logo';
				$attr['itemscope'] = 'itemscope';
				$attr['itemtype']  = 'https://schema.org/Organization ';
				break;

			case 'name':
				$attr['itemprop'] = 'name';
				break;

			case 'url':
				$attr['itemprop'] = 'url';
				break;

			case 'email':
				$attr['itemprop'] = 'email';
				break;

			case 'job':
				$attr['itemprop'] = 'jobTitle';
				break;

			case 'post_time':
				$attr['itemprop'] = 'datePublished';
				$attr['datetime'] = get_the_time( 'c', $args['id'] );
				break;

			case 'post_title':
				$attr['itemprop'] = 'headline';
				break;

			case 'post_content':
				$attr['itemprop'] = 'text';
				break;
		}

		foreach ( $attr as $key => $value ) {
			$attributes .= $key . '="' . $value . '" ';
		}

		echo wp_kses( $attributes, azo_get_allowed_html() );
	}
}

/**
 * Function alllowed html
 */
function azo_shape_space_allowed_html() {
	$allowed_tags = array(
		'a'          => array(
			'class' => array(),
			'href'  => array(),
			'rel'   => array(),
			'title' => array(),
		),
		'abbr'       => array(
			'title' => array(),
		),
		'b'          => array(),
		'blockquote' => array(
			'cite' => array(),
		),
		'cite'       => array(
			'title' => array(),
		),
		'code'       => array(),
		'del'        => array(
			'datetime' => array(),
			'title'    => array(),
		),
		'dd'         => array(),
		'div'        => array(
			'class' => array(),
			'title' => array(),
			'style' => array(),
		),
		'dl'         => array(),
		'dt'         => array(),
		'em'         => array(),
		'h1'         => array(),
		'h2'         => array(),
		'h3'         => array(),
		'h4'         => array(),
		'h5'         => array(),
		'h6'         => array(),
		'i'          => array(),
		'img'        => array(
			'alt'    => array(),
			'class'  => array(),
			'height' => array(),
			'src'    => array(),
			'width'  => array(),
		),
		'li'         => array(
			'class' => array(),
		),
		'ol'         => array(
			'class' => array(),
		),
		'p'          => array(
			'class' => array(),
		),
		'q'          => array(
			'cite'  => array(),
			'title' => array(),
		),
		'span'       => array(
			'class' => array(),
			'title' => array(),
			'style' => array(),
		),
		'strike'     => array(),
		'strong'     => array(),
		'ul'         => array(
			'class' => array(),
		),
	);
	return $allowed_tags;
}

if ( ! function_exists( 'azo_get_allowed_html' ) ) {
	/**
	 * Return allowed html tags
	 *
	 * @since 1.0.0
	 *
	 * @return array
	 */
	function azo_get_allowed_html() {
		return apply_filters(
			'azo_allowed_html',
			array(
				'br'     => array(),
				'i'      => array(),
				'b'      => array(),
				'u'      => array(),
				'em'     => array(),
				'del'    => array(),
				'a'      => array(
					'href'  => true,
					'class' => true,
					'title' => true,
					'rel'   => true,
				),
				'strong' => array(),
				'span'   => array(
					'style' => true,
					'class' => true,
				),
			)
		);
	}
}

/**
 * ------------------------------------------------------------------------------------------------
 * is ajax request
 * ------------------------------------------------------------------------------------------------
 */

if ( ! function_exists( 'azo_is_woo_ajax' ) ) {
	/**
	 * Azo is woo ajax
	 */
	function azo_is_woo_ajax() {
		$request_headers = function_exists( 'getallheaders' ) ? getallheaders() : array();

		if ( azo_is_elementor_installed() && azo_elementor_is_edit_mode() ) {
			return apply_filters( 'xts_is_ajax', false );
		}

		if ( defined( 'DOING_AJAX' ) && DOING_AJAX ) {
			return 'wp-ajax';
		}

		if ( isset( $request_headers['x-pjax'] ) || isset( $request_headers['X-PJAX'] ) || isset( $request_headers['X-Pjax'] ) ) {
			return 'full-page';
		}

		if ( isset( $_REQUEST['woo_ajax'] ) ) {
			return 'fragments';
		}

		if ( azo_is_pjax() ) {
			return true;
		}

		return false;
	}
}

if ( ! function_exists( 'azo_is_pjax' ) ) {
	/**
	 * Azo is pjax
	 */
	function azo_is_pjax() {
		$request_headers = function_exists( 'getallheaders' ) ? getallheaders() : array();

		//return isset( $_REQUEST['_pjax'] ) && ( ( isset( $request_headers['X-Requested-With'] ) && 'xmlhttprequest' === strtolower( $request_headers['X-Requested-With'] ) ) || ( isset( $_SERVER['HTTP_X_REQUESTED_WITH'] ) && 'xmlhttprequest' === strtolower( $_SERVER['HTTP_X_REQUESTED_WITH'] ) ) );
	}
}

if ( ! function_exists( 'azo_is_elementor_installed' ) ) {
	/**
	 * Check if Elementor is activated
	 *
	 * @since 1.0.0
	 * @return boolean
	 */
	function azo_is_elementor_installed() {
		return did_action( 'elementor/loaded' ) && 'elementor' === azo_get_opt( 'page_builder', 'wpb' );
	}
}

if ( ! function_exists( 'azo_get_old_classes' ) ) {
	/**
	 * Get old classes.
	 *
	 * @since 1.0.0
	 *
	 * @param string $classes Classes.
	 *
	 * @return string
	 */
	function azo_get_old_classes( $classes ) {
		if ( ! azo_get_opt( 'old_elements_classes', true ) ) {
			$classes = '';
		}

		return esc_html( $classes );
	}
}

if ( ! function_exists( 'azo_seo_data' ) ) {
	/**
	 * SCHEMA MARKUP FOR SEO
	 */
	function azo_seo_data() {
		?>
		<div class="screen-reader-text">
			<h2 <?php azo_schema_markup( 'title' ); ?>>
				<?php echo esc_html( get_the_title() ); ?>
			</h2>

			<?php /*THUMBNAIL*/ ?>
			<?php
			global $post;
			$thumbnail = get_the_post_thumbnail_url( $post->ID, 'thumbnail' );

			if ( ! empty( $thumbnail ) ) {
				$img_id  = get_post_thumbnail_id( $post->ID );
				$img_alt = azo_img_alt( $img_id, esc_attr__( 'Blog thumbnail', 'azo' ) );
				?>
				<span <?php azo_schema_markup( 'image' ); ?>>
						<img src="<?php echo esc_url( $thumbnail ); ?>" alt="<?php echo esc_attr( $img_alt ); ?>">
						<meta itemprop="url" content="<?php echo esc_url( $thumbnail ); ?>"/>
						<meta itemprop="width" content="770"/>
						<meta itemprop="height" content="450"/>
					</span>
			<?php } else { ?>
				<span <?php azo_schema_markup( 'image' ); ?>>
					<img src="<?php echo esc_url( get_template_directory_uri() . '/images/thumbnail-default.jpg' ); ?>" alt="<?php esc_attr_e( 'Thumbnail', 'azo' ); ?>">
					<meta itemprop="url" content="<?php echo esc_url( get_template_directory_uri() . '/images/thumbnail-default.jpg' ); ?>"/>
					<meta itemprop="width" content="100"/>
					<meta itemprop="height" content="90"/>
				</span>
			<?php } ?>

			<?php /*PUBLISHER*/ ?>
			<span class="author" <?php azo_schema_markup( 'publisher' ); ?>>
				<span itemprop="logo" itemscope="itemscope" itemtype="https://schema.org/ImageObject">
					<meta itemprop="url" content="<?php echo esc_url( home_url( '/' ) ); ?>"/>
					<meta itemprop="width" content="100"/>
					<meta itemprop="height" content="100"/>
				</span>
				<meta itemprop="name" content="<?php the_author(); ?>"/>
			</span>

			<?php /*DATE MODIFIED*/ ?>
			<span itemprop="dateModified" class="updated">
				<time datetime="<?php echo esc_attr( get_the_modified_time( 'Y-m-d' ) ); ?>">
					<?php the_modified_date(); ?>
				</time>
			</span>

			<?php /*DATE PUBLISHED*/ ?>
			<span itemprop="datePublished" content="<?php echo esc_attr( get_the_time( 'c' ) ); ?>">
				<?php echo esc_html( get_the_time( 'c' ) ); ?>
			</span>

			<?php /*POST CATEGORIES*/ ?>
			<span itemprop="articleSection">
				<?php echo wp_kses( azo_blog_categories(), azo_get_allowed_html() ); ?>
			</span>
		</div>
		<?php
	}
}

if ( ! function_exists( 'azo_wp_link_pages' ) ) {
	/**
	 * THEME PAGINATION ( SINGLE POST/PAGE )
	 */
	function azo_wp_link_pages() {
		$args = array(
			'before'      => '<div class="theme-page-links">',
			'after'       => '</div>',
			'link_before' => '<span>',
			'link_after'  => '</span>',
		);

		return wp_link_pages( $args );
	}
}

if ( ! function_exists( 'azo_pagination' ) ) {
	/**
	 * Azo pagination
	 *
	 * @param  [type] $total_page type.
	 */
	function azo_pagination( $total_page ) {

		if ( $total_page > 1 ) {
			$current_page = max( 1, get_query_var( 'paged' ) );

			echo '<nav class="azo-pagination blog-pagination pagination">';

			echo paginate_links( // phpcs:ignore
				array(
					'base'      => get_pagenum_link( 1 ) . '%_%',
					'format'    => '/page/%#%',
					'current'   => $current_page,
					'total'     => $total_page,
					'prev_text' => '',
					'next_text' => '',
					'end_size'  => 3,
					'mid_size'  => 3,
				)
			);

			echo '</nav>';
		}
	}
}

add_filter( 'wp_generate_tag_cloud', 'azo_tag_cloud', 10, 1 );
/**
 * REMOVE TAG-CLOUD INLINE CSS
 *
 * @param  [type] $string TYPE.
 */
function azo_tag_cloud( $string ) {
	return preg_replace( '/ style=("|\')(.*?)("|\')/', '', $string );
}

if ( ! function_exists( 'azo_dialog_search_form' ) ) {
	/**
	 * THEME SEARCH FORM
	 */
	function azo_dialog_search_form() {
		?>
		<div class="azo-search-form-container js-search-form">
			<div class="container azo-search-form__inner">
				<form action="<?php echo esc_url( home_url( '/' ) ); ?>" method="get" class="azo-search-form">
					<input
							type="text"
							name="s"
							class="azo-search-form__field js-search-field"
							placeholder="<?php esc_attr_e( 'Search...', 'azo' ); ?>"
							autocomplete="off"
					>
					<?php
					$product_only = get_theme_mod( 'product_search', false );
					if ( $product_only ) {
						if ( get_theme_mod( 'product_cat_in_search', true ) ) {
							azo_dropdown_product_categories();
						}
						?>
						<input type="hidden" name="post_type" value="product">
						<?php
					}
					?>
					<span class="azo-search-form__desc"><?php esc_html_e( 'Hit Enter to search or Esc key to close', 'azo' ); ?></span>
					<a href="#" class="azo-search-form__close js-close-search-form">
						<span class="ion-android-close"></span>
					</a>
				</form><!-- .ajax-search-form -->
			</div><!-- .ajax-search-form__inner -->
		</div><!-- .ajax-search-form-container -->
		<?php
	}
}

if ( ! function_exists( 'azo_dropdown_product_categories' ) ) {
	/**
	 * Dropdown product categories
	 */
	function azo_dropdown_product_categories() {
		if ( isset ( $_GET['product_cat'] ) && ! empty ( $_GET['product_cat'] ) ) { // phpcs:ignore
			$opt_select = sanitize_text_field( wp_unslash( $_REQUEST['product_cat'] ) );
		} else {
			$opt_select = 0;
		}
		$args = array(
			'show_option_all' => esc_html__( 'All Categories', 'azo' ),
			'hierarchical'    => 1,
			'class'           => 'cat',
			'echo'            => 1,
			'value_field'     => 'slug',
			'selected'        => $opt_select,
		);

		$args['taxonomy'] = 'product_cat';
		$args['name']     = 'product_cat';
		$args['class']    = 'cate-dropdown hidden-xs';
		wp_dropdown_categories( $args );
	}
}

if ( ! function_exists( 'azo_ajax_search_form' ) ) {
	/**
	 * Ajax search form
	 */
	function azo_ajax_search_form() {
		?>
		<div class="azo-search-form-container js-search-form">
			<div class="container azo-search-form__inner">
				<form action="<?php echo esc_url( home_url( '/' ) ); ?>" method="get" class="azo-search-form">
					<input
							type="text"
							name="s"
							id="autocomplete"
							class="azo-search-form__field js-search-field"
							placeholder="<?php esc_attr_e( 'Search Products...', 'azo' ); ?>"
							autocomplete="off"
					>
					<?php
					if ( get_theme_mod( 'product_cat_in_search', true ) ) {
						azo_dropdown_product_categories();
					}
					?>
					<input type="hidden" name="post_type" value="product">
					<span class="fa fa-circle-o-notch fa-2x fa-fw azo-search-form__loader"></span>
					<a href="#" class="azo-search-form__close js-close-search-form">
						<span class="ion-android-close"></span>
					</a>
				</form><!-- .ajax-search-form -->

				<div class="azo-search-form__suggestions"></div>
			</div><!-- .ajax-search-form__inner -->
		</div><!-- .ajax-search-form-container -->
		<?php
	}
}

if ( ! function_exists( 'azo_ie' ) ) {
	/**
	 * DETECT IE BROWSER
	 */
	function azo_ie() {
		global $is_winIE;

		if ( $is_winIE ) {
			return true;
		}

		return false;
	}
}

if ( ! function_exists( 'azo_search_form' ) ) {
	/**
	 * Azo search form
	 */
	function azo_search_form() {
		if ( get_theme_mod( 'ajax_search', false ) ) {
			azo_ajax_search_form();
		} else {
			azo_dialog_search_form();
		}
	}
}

// BLOG.
if ( ! function_exists( 'azo_blog' ) ) {
	/**
	 * DETECT BLOG PAGE
	 */
	function azo_blog() {
		global $post;
		$post_type = get_post_type( $post );

		return ( 'post' === $post_type && ( is_archive() || is_author() || is_category() || is_home() || is_single() || is_tag() ) ) ? true : false;
	}
}

/**
 * Filter the except length to words.
 *
 * @param int $length Excerpt length.
 * @return int (Maybe) modified excerpt length.
 */
function azo_custom_excerpt_length( $length ) {
	$length = get_theme_mod( 'excerpt_length', 32 );

	return $length;
}
add_filter( 'excerpt_length', 'azo_custom_excerpt_length', 999 );

if ( ! function_exists( 'azo_blog_tags' ) ) {
	/**
	 * BLOG TAGS
	 */
	function azo_blog_tags() {
		if ( false !== get_theme_mod( 'show_tag', true ) ) {
			if ( has_tag() ) {
				?>
				<span class="tagcloud" itemprop="keywords">
					<span class="name-tag">
						<?php esc_html_e( 'Tags:', 'azo' ); ?>
					</span>
					<?php echo wp_kses( the_tags( ' ' ), azo_get_allowed_html() ); ?>
				</span>
				<?php
			}
		}
	}
}

/**
 * Azo get sharing list
 */
function azo_get_sharing_list() {
	$social_sharing_order = get_theme_mod( 'social_sharing_order', array( 'facebook', 'twitter', 'pinterest' ) );

	foreach ( $social_sharing_order as $social ) {
		if ( 'facebook' === $social ) {
			?>
			<li class="list-inline-item">
				<a href="https://www.facebook.com/sharer.php?u=<?php echo rawurlencode( get_permalink() ); ?>" class="social social-facebook">
					<i class="fa fa-facebook"></i>
				</a>
			</li>
		<?php } elseif ( 'twitter' === $social ) { ?>
			<li class="list-inline-item">
				<a href="https://twitter.com/share?text=<?php echo rawurlencode( html_entity_decode( get_the_title(), ENT_COMPAT, 'UTF-8' ) ); ?>&url=<?php echo rawurlencode( get_permalink() ); ?>" class="social social-twitter">
					<i class="fa fa-twitter"></i>
				</a>
			</li>
		<?php } elseif ( 'pinterest' === $social ) { ?>
			<li class="list-inline-item">
				<a href="https://pinterest.com/pin/create/link/?url=<?php echo rawurlencode( get_permalink() ); ?>&amp;description=<?php echo rawurlencode( get_the_title() ); ?>" class="social social-pinterest">
					<i class="fa fa-pinterest-p"></i>
				</a>
			</li>
		<?php } elseif ( 'tumblr' === $social ) { ?>
			<li class="list-inline-item">
				<a href="https://www.tumblr.com/share/link?url=<?php echo rawurlencode( get_permalink() ); ?>&amp;name=<?php echo rawurlencode( get_the_title() ); ?>" class="social social-tumblr">
					<i class="fa fa-tumblr-square"></i>
				</a>
			</li>
		<?php } elseif ( 'linkedin' === $social ) { ?>
			<li class="list-inline-item">
				<a href="https://www.linkedin.com/shareArticle?mini=true&url=<?php echo rawurlencode( get_permalink() ); ?>&amp;title=<?php echo rawurlencode( get_the_title() ); ?>" class="social social-linkedin">
					<i class="fab fa-linkedin"></i>
				</a>
			</li>
		<?php } elseif ( 'email' === $social ) { ?>
			<li class="list-inline-item">
				<a href="mailto:?subject=<?php echo rawurlencode( get_the_title() ); ?>&amp;body=<?php echo rawurlencode( get_permalink() ); ?>" class="social social-email">
					<i class="fas fa-envelope"></i>
				</a>
			</li>
			<?php
		}
	}
}

/**
 * Azo share social
 */
function azo_share_social() {
	if ( true === get_theme_mod( 'show_share', false ) ) {
		?>
		<div class="share-social">
			<div class="title-share">
				<?php echo esc_html__( 'Share:', 'azo' ); ?>
			</div>
			<ul class="social-list">
				<?php azo_get_sharing_list(); ?>
			</ul>
		</div>
		<?php
	}
}

/**
 * Azo get shop sharing list
 */
function azo_get_shop_sharing_list() {
	$social_shop_sharing_order = get_theme_mod( 'social_shop_sharing_order', array( 'facebook', 'twitter', 'pinterest' ) );

	foreach ( $social_shop_sharing_order as $social ) {
		if ( 'facebook' === $social ) {
			?>
			<a href="https://www.facebook.com/sharer.php?u=<?php echo rawurlencode( get_permalink() ); ?>" class="social social-facebook">
				<i class="fa fa-facebook"></i>
			</a>
		<?php } elseif ( 'twitter' === $social ) { ?>
			<a href="https://twitter.com/share?text=<?php echo rawurlencode( html_entity_decode( get_the_title(), ENT_COMPAT, 'UTF-8' ) ); ?>&url=<?php echo rawurlencode( get_permalink() ); ?>" class="social social-twitter">
				<i class="fa fa-twitter"></i>
			</a>
		<?php } elseif ( 'pinterest' === $social ) { ?>
			<a href="https://pinterest.com/pin/create/link/?url=<?php echo rawurlencode( get_permalink() ); ?>&amp;description=<?php echo rawurlencode( get_the_title() ); ?>" class="social social-pinterest">
				<i class="fa fa-pinterest-p"></i>
			</a>
		<?php } elseif ( 'tumblr' === $social ) { ?>
			<a href="https://www.tumblr.com/share/link?url=<?php echo rawurlencode( get_permalink() ); ?>&amp;name=<?php echo rawurlencode( get_the_title() ); ?>" class="social social-tumblr">
				<i class="fa fa-tumblr-square"></i>
			</a>
		<?php } elseif ( 'linkedin' === $social ) { ?>
			<a href="https://www.linkedin.com/shareArticle?mini=true&url=<?php echo rawurlencode( get_permalink() ); ?>&amp;title=<?php echo rawurlencode( get_the_title() ); ?>" class="social social-linkedin">
				<i class="fab fa-linkedin"></i>
			</a>
		<?php } elseif ( 'email' === $social ) { ?>
			<a href="mailto:?subject=<?php echo rawurlencode( get_the_title() ); ?>&amp;body=<?php echo rawurlencode( get_permalink() ); ?>" class="social social-email">
				<i class="fas fa-envelope"></i>
			</a>
			<?php
		}
	}
}

if ( ! function_exists( 'azo_paging' ) ) {
	/**
	 * BLOG PAGINATION
	 *
	 * @param  [type] $wp_query type.
	 * @param  [type] $paged    type.
	 */
	function azo_paging( $wp_query = null, $paged = null ) {

		if ( ! $wp_query ) {
			$wp_query = $GLOBALS['wp_query'];
		}

		/*Don't print empty markup if there's only one page.*/

		if ( $wp_query->max_num_pages < 2 ) {
			return;
		}

		if ( null === $paged ) {
			$paged = get_query_var( 'paged' ) ? intval( get_query_var( 'paged' ) ) : 1;
		}
		$pagenum_link = html_entity_decode( get_pagenum_link() );
		$query_args   = array();
		$url_parts    = explode( '?', $pagenum_link );

		if ( isset( $url_parts[1] ) ) {
			wp_parse_str( $url_parts[1], $query_args );
		}

		$pagenum_link = remove_query_arg( array_keys( $query_args ), $pagenum_link );
		$pagenum_link = trailingslashit( $pagenum_link ) . '%_%';

		$format  = $GLOBALS['wp_rewrite']->using_index_permalinks() && ! strpos( $pagenum_link, 'index.php' ) ? 'index.php/' : '';
		$format .= $GLOBALS['wp_rewrite']->using_permalinks() ? user_trailingslashit( 'page/%#%', 'paged' ) : '?paged=%#%';

		/*Set up paginated links.*/
		$links = paginate_links(
			array(
				'base'      => $pagenum_link,
				'format'    => $format,
				'total'     => $wp_query->max_num_pages,
				'current'   => $paged,
				'mid_size'  => 1,
				'add_args'  => array_map( 'urlencode', $query_args ),
				'prev_text' => sprintf( esc_html_x( '%s', 'Pagination previous button', 'azo' ), '<span class="fa fa-angle-double-left"></span>' ),
				'next_text' => sprintf( esc_html_x( '%s', 'Pagination next button', 'azo' ), '<span class="fa fa-angle-double-right"></span>' ),
				'type'      => 'list',
			)
		);

		$blog_preset = get_theme_mod( 'blog_preset', 'blog1' );

		if ( 'blog1' === $blog_preset ) {
			$class = 'ht-pagination';
		} else {
			$class = 'azo-pagination blog-pagination ht-pagination';
		}

		if ( $links ) {
			?>
			<nav class="<?php echo esc_attr( $class ); ?>">
				<span class="screen-reader-text"><?php esc_html_e( 'Posts pagination', 'azo' ); ?></span>
				<?php echo wp_kses( $links, azo_get_allowed_html() ); ?>
			</nav>
			<?php
		}
	}
}

if ( ! function_exists( 'azo_blog_categories' ) ) {
	/**
	 * BLOG CATEGORIES
	 */
	function azo_blog_categories() {
		return get_the_term_list(
			get_the_ID(),
			'category',
			esc_html_x( '', 'In Uncategorized Category', 'azo' ),
			', ',
			null
		);
	}
}

if ( ! function_exists( 'azo_blog_date' ) ) {
	/**
	 * BLOG DATE
	 */
	function azo_blog_date() {
		?>
		<span class="azo-date-time">
			<time datetime="<?php echo esc_attr( get_the_modified_time( 'Y-m-d' ) ); ?>">
				<?php the_modified_date(); ?>
			</time>
		</span>
		<?php
	}
}

if ( ! function_exists( 'azo_comment_list' ) ) {
	/**
	 * COMMENT LIST MODIFIED
	 *
	 * @param  [type] $comment TYPE.
	 * @param  [type] $args    TYPE.
	 * @param  [type] $depth   TYPE.
	 */
	function azo_comment_list( $comment, $args, $depth ) {
		$GLOBALS['comment'] = $comment;

		switch ( $comment->comment_type ) :
			case 'pingback':
			case 'trackback':
				?>
				<div class="comment-post-pingback">
					<span><?php esc_html_e( 'Pingback:', 'azo' ); ?></span><?php comment_author_link(); ?><?php edit_comment_link( esc_html__( 'Edit', 'azo' ) ); ?>
				</div>
				<?php
				break;
			default:
				?>
				<div id="comment-<?php comment_ID(); ?>" class="comment-item">
					<div class="comment-avatar">
						<?php echo get_avatar( $comment, $size = 100 ); ?>
					</div>
					<div class="comment-content">
						<strong class="comment-author">
							<a href="#comment-<?php comment_ID(); ?>" class="comment-author-name">
								<?php echo esc_html( get_comment_author() ); ?>
							</a>
							<span class="comment-time">
								<?php echo esc_html( azo_date_new_format() ); ?>
							</span>
						</strong>
						<div class="comment-info">
							<?php
							comment_reply_link(
								array_merge(
									$args,
									array(
										'depth'      => $depth,
										'max_depth'  => $args['max_depth'],
										'reply_text' => esc_html__( 'Reply', 'azo' ),
									)
								)
							);

							edit_comment_link( esc_html__( 'Edit', 'azo' ), ' ', '' );
							?>
						</div>
						<div class="comment-text">
							<?php if ( '0' === $comment->comment_approved ) { ?>
								<em>
									<?php esc_html_e( 'Your comment is awaiting moderation.', 'azo' ); ?>
								</em>
							<?php } ?>
							<?php comment_text(); ?>
						</div>
					</div>
				</div>
				<?php
				break;
		endswitch;
	}
}

if ( ! function_exists( 'azo_date_new_format' ) ) {
	/**
	 * GET DATE TIME
	 */
	function azo_date_new_format() {
		$date_format = get_the_date( get_option( 'date_format' ) );

		return $date_format;
	}
}

if ( ! function_exists( 'azo_date_format' ) ) {
	/**
	 * GET DATE TIME
	 */
	function azo_date_format() {
		?>
		<span class="day">
			<?php echo esc_html( get_the_modified_date( 'j' ) ); ?>
		</span>

		<span class="month">
			<?php echo esc_html( get_the_modified_date( 'M' ) ); ?>
		</span>
		<?php
	}
}

if ( ! function_exists( 'azo_post_info' ) ) {
	/**
	 * BLOG POST INFO
	 */
	function azo_post_info() {
		global $post;
		?>
		<span class="entry-meta blog-header-info">
			<?php if ( false !== get_theme_mod( 'author_blog', true ) ) { ?>
				<span class="if-item if-author" itemprop="author">
					<?php
					$author_id = get_post_field( 'post_author' );
					echo get_avatar( $author_id, 40 );

					the_author_posts_link();
					?>
				</span>
			<?php } ?>

			<?php if ( false !== azo_blog_categories() && false !== get_theme_mod( 'category_blog', true ) ) { ?>
				<span class="if-item if-cat">
					<?php echo wp_kses( azo_blog_categories(), azo_get_allowed_html() ); ?>
				</span>
			<?php } ?>

			<?php
			if ( false !== get_theme_mod( 'count_comment_blog', true ) ) {
				azo_comment_count();
			}
			?>
		</span>
		<?php
	}
}

if ( ! function_exists( 'azo_single_post_info' ) ) {
	/**
	 * BLOG POST INFO
	 */
	function azo_single_post_info() {
		global $post;
		?>
		<span class="entry-meta blog-header-info">
			<?php if ( false !== get_theme_mod( 'author_blog', true ) ) { ?>
				<span class="if-item if-author" itemprop="author">
					<?php
					$author_id = get_post_field( 'post_author' );
					echo get_avatar( $author_id, 40 );

					the_author_posts_link();
					?>
				</span>
			<?php } ?>

			<?php
			if ( false !== get_theme_mod( 'date_time_info', true ) ) {
				global $post;
				$thumbnail = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ) );

				if ( empty( $thumbnail ) ) {
					?>
					<span class="if-item" itemprop="datePublished" datetime="<?php echo esc_attr( get_the_time( 'c' ) ); ?>">
						<?php echo esc_html( azo_date_new_format() ); ?>
					</span>
					<?php
				}
			}
			?>

			<?php if ( false !== azo_blog_categories() && false !== get_theme_mod( 'category_blog_single', true ) ) { ?>
				<span class="if-item if-cat">
					<?php echo wp_kses( azo_blog_categories(), azo_get_allowed_html() ); ?>
				</span>
			<?php } ?>

			<?php
			if ( false !== get_theme_mod( 'count_comment_blog_single', true ) ) {
				azo_comment_count();
			}
			?>
		</span>
		<?php
	}
}

if ( ! function_exists( 'azo_blog_widget_info' ) ) {
	/**
	 * BLOG WIDGET INFO
	 */
	function azo_blog_widget_info() {
		global $post;
		?>
		<span class="entry-meta blog-header-info">
			<span class="if-item if-cat">
				<?php echo wp_kses( azo_blog_categories(), azo_get_allowed_html() ); ?>
			</span>

			<time class="if-item if-date" itemprop="datePublished" datetime="<?php echo esc_attr( get_the_time( 'c' ) ); ?>">
				<?php echo esc_html( azo_date_format() ); ?>
			</time>			

			<?php azo_comment_count(); ?>
		</span>
		<?php
	}
}

if ( ! function_exists( 'azo_comment_count' ) ) :
	/**
	 * Prints HTML with the comment count for the current post.
	 */
	function azo_comment_count() {
		if ( ! post_password_required() && ( comments_open() || get_comments_number() ) ) {
			echo '<span class="if-item if-comment">';

			/* translators: %s: Post title. Only visible to screen readers. */
			comments_popup_link();

			echo '</span>';
		}
	}
endif;

if ( ! function_exists( 'azo_blog_widget_all_author' ) ) {
	/**
	 * BLOG WIDGET ALL AUTHOR
	 */
	function azo_blog_widget_all_author() {
		global $post;
		?>
		<span class="entry-meta blog-header-info">
			<span class="if-item if-cat">
				<?php echo wp_kses( azo_blog_categories(), azo_get_allowed_html() ); ?>
			</span>

			<time class="if-item if-date" itemprop="datePublished" datetime="<?php echo esc_attr( get_the_time( 'c' ) ); ?>">
				<?php echo esc_html( azo_date_format() ); ?>
			</time>			

			<span class="if-item if-author" itemprop="author">
				<?php
				esc_html_e( 'by ', 'azo' );
				$author_id = get_post_field( 'post_author' );
				the_author_posts_link();
				?>
			</span>
		</span>
		<?php
	}
}


if ( ! function_exists( 'azo_blogwidget_allauthor' ) ) {
	/**
	 * BLOG WIDGET AUTHOR, DATE, COMMENT
	 */
	function azo_blogwidget_allauthor() {
		global $post;
		?>
		<span class="entry-meta blog-header-info">
			<span class="if-item if-author" itemprop="author">
				<?php
				esc_html_e( 'by ', 'azo' );
				$author_id = get_post_field( 'post_author' );
				the_author_posts_link();
				?>
			</span>

			<time class="if-item if-date" itemprop="datePublished" datetime="<?php echo esc_attr( get_the_time( 'c' ) ); ?>">
				<?php echo esc_html( azo_date_format() ); ?>
			</time>

			<?php azo_comment_count(); ?>
		</span>
		<?php
	}
}

if ( ! function_exists( 'azo_blog_widget_date_author' ) ) {
	/**
	 * BLOG WIDGET DATE AUTHOR AVATAR
	 */
	function azo_blog_widget_date_author() {
		global $post;
		?>
		<span class="entry-meta blog-header-info">
			<span class="if-item if-author" itemprop="author">
				<?php
				$author_id = get_post_field( 'post_author' );
				echo get_avatar( $author_id, 40 );

				the_author_posts_link();
				?>
			</span>

			<time class="if-item if-date" itemprop="datePublished" datetime="<?php echo esc_attr( get_the_time( 'c' ) ); ?>">
				<?php echo esc_html( azo_date_format() ); ?>
			</time>			
		</span>
		<?php
	}
}

if ( ! function_exists( 'azo_post_thumbnail' ) ) {
	/**
	 * Post thumbnail
	 *
	 * @param  string $size type.
	 */
	function azo_post_thumbnail( $size = 'full' ) {
		global $post;
		$thumbnail = wp_get_attachment_image_src( get_post_thumbnail_id( $post->ID ), $size );
		if ( empty( $thumbnail ) ) {
			return;
		}

		$img_id  = get_post_thumbnail_id( $post->ID );
		$img_alt = azo_img_alt( $img_id, esc_attr__( 'Blog thumbnail', 'azo' ) );

		if ( is_single() ) {
			?>
			<div class="cover-image" <?php azo_schema_markup( 'image' ); ?>>
				<?php if ( false !== get_theme_mod( 'date_time_info_single', true ) ) { ?>
					<time class="if-item if-date" itemprop="datePublished" datetime="<?php echo esc_attr( get_the_time( 'c' ) ); ?>">
						<?php echo esc_html( azo_date_format() ); ?>
					</time>
				<?php } ?>

				<img src="<?php echo esc_url( $thumbnail[0] ); ?>" alt="<?php echo esc_attr( $img_alt ); ?>">
				<meta itemprop="url" content="<?php echo esc_url( $thumbnail[0] ); ?>"/>
				<meta itemprop="width" content="<?php echo esc_attr( $thumbnail[1] ); ?>"/>
				<meta itemprop="height" content="<?php echo esc_attr( $thumbnail[2] ); ?>"/>
			</div>
		<?php } else { ?>
			<a href="<?php the_permalink(); ?>" class="cover-image entry-image-link" <?php azo_schema_markup( 'image' ); ?>>
				<?php if ( false !== get_theme_mod( 'date_time_info_single', true ) ) { ?>
					<time class="if-item if-date" itemprop="datePublished" datetime="<?php echo esc_attr( get_the_time( 'c' ) ); ?>">
						<?php echo esc_html( azo_date_format() ); ?>
					</time>
				<?php } ?>

				<img src="<?php echo esc_url( $thumbnail[0] ); ?>" alt="<?php echo esc_attr( $img_alt ); ?>">
				<meta itemprop="url" content="<?php echo esc_url( $thumbnail[0] ); ?>"/>
				<meta itemprop="width" content="<?php echo esc_attr( $thumbnail[1] ); ?>"/>
				<meta itemprop="height" content="<?php echo esc_attr( $thumbnail[2] ); ?>"/>
				<?php if ( is_sticky() ) : ?>
					<span class="sticky-medal"><span><?php esc_html_e( 'Sticky', 'azo' ); ?></span></span>
				<?php endif; ?>
			</a>
			<?php
		}
	}
}

if ( ! function_exists( 'azo_post_format' ) ) {
	/**
	 * POST FORMAT
	 */
	function azo_post_format() {
		global $post;

		azo_post_thumbnail();
	}
}

add_filter( 'get_search_form', 'azo_search_form_widget', 100 );
/**
 * OUTPUT SEARCH WIDGET MODIFIED
 *
 * @param  [type] $form type.
 */
function azo_search_form_widget( $form ) {
	$form  = '<form role="search" method="get" class="search-form" action="' . esc_url( home_url( '/' ) ) . '" >';
	$form .= '<label class="screen-reader-text">' . esc_html__( 'Search for:', 'azo' ) . '</label>';
	$form .= '<input type="text" class="search-field" placeholder="' . esc_attr__( 'Search...', 'azo' ) . '" value="' . get_search_query() . '" name="s" required/>';
	$form .= '<button type="submit" class="search-submit icoros-icon_search"></button>';
	$form .= '</form>';

	return $form;
}

add_filter( 'excerpt_more', 'azo_more' );
/**
 * CUSTOM EXERPT MORE TEXT
 *
 * @param  [type] $more type.
 */
function azo_more( $more ) {
	return '...';
}


// PLUGINS.
if ( ! function_exists( 'azo_custom_font' ) ) {
	/**
	 * CUSTOM FONT-FAMILY FOR KIRKI
	 */
	function azo_custom_font() {
		$fonts = array(
			'fonts' => array(
				'families' => array(
					'custom' => array(
						'text'     => esc_html__( 'azo custom font', 'azo' ),
						'children' => array(
							array(
								'id'   => 'Rajdhani',
								'text' => esc_html__( 'Rajdhani', 'azo' ),
							),
						),
					),
				),
				'variants' => array(
					'Rajdhani' => array(
						'300',
						'regular',
						'500',
						'600',
						'700',
					),
				),
			),
		);

		return $fonts;
	}
}

if ( is_admin() ) {
	add_action( 'registered_post_type', 'azo_remove_rev_slider_meta_boxes' );
	/**
	 * REMOVE REV SLIDER METABOX
	 *
	 * @param  [type] $post_type type.
	 */
	function azo_remove_rev_slider_meta_boxes( $post_type ) {
		add_action(
			'do_meta_boxes',
			function () use ( $post_type ) {
				remove_meta_box( 'mymetabox_revslider_0', $post_type, 'normal' );
			}
		);
	}
}

/**
 * REMOVE CONTACT FORM 7 CSS
 */
add_filter( 'wpcf7_load_css', '__return_false' );

/**
 * DISABLE THE CONFIGURATION CTF7 VALIDATOR
 */
add_filter( 'wpcf7_validate_configuration', '__return_false' );

// ELEMENTOR.
if ( ! function_exists( 'azo_is_elementor' ) ) {
	/**
	 * CHECK IF ELEMENTOR IS ACTIVE
	 */
	function azo_is_elementor() {
		return defined( 'ELEMENTOR_VERSION' );
	}
}

if ( ! function_exists( 'azo_is_elementor_pro' ) ) {
	/**
	 * Check elementor pro
	 */
	function azo_is_elementor_pro() {
		return defined( 'ELEMENTOR_PRO_VERSION' );
	}
}

if ( ! function_exists( 'azo_elementor_page' ) ) {
	/**
	 * CHECK IF PAGE BUILD WITH ELEMENTOR
	 *
	 * @param  [type] $id type.
	 */
	function azo_elementor_page( $id ) {
		return get_post_meta( $id, '_elementor_edit_mode', true );
	}
}

if ( ! function_exists( 'azo_page_opt' ) ) {
	/**
	 * GET PAGE OPTION
	 *
	 * @param  [type] $_id type.
	 * @param  string $key type.
	 */
	function azo_page_opt( $_id = null, $key = '' ) {
		if ( true === azo_is_elementor() && true === azo_elementor_page( $_id ) ) {
			$document = \Elementor\Plugin::$instance->documents->get( $_id );

			if ( $document ) {
				return $document->get_settings( $key );
			}

			return false;
		}
	}
}

if ( ! function_exists( 'azo_get_narrow_data' ) ) {
	/**
	 * TOURS WIDGET: GET NARROW DATA SOURCE
	 *
	 * @param  string $type  type.
	 * @param  string $terms type.
	 */
	function azo_get_narrow_data( $type = 'post', $terms = 'category' ) {
		$output = array();
		switch ( $type ) :
			case 'post':
				$tour_args = array(
					'post_type'           => $terms,
					'post_status'         => 'publish',
					'ignore_sticky_posts' => 1,
					'posts_per_page'      => - 1,
				);
				$qr        = new WP_Query( $tour_args );
				$output    = wp_list_pluck( $qr->posts, 'post_title', 'ID' );
				break;

			case 'term':
				$terms  = get_terms( $terms );
				$output = wp_list_pluck( $terms, 'name', 'term_id' );
				break;
		endswitch;

		return $output;
	}
}

// Add custom markup for Elementor Canvas header.
add_action( 'elementor/page_templates/canvas/before_content', 'azo_preloader' );

add_action( 'elementor/page_templates/canvas/after_content', 'azo_elementor_canvas_footer' );
/**
 * Add custom markup for Elementor Canvas footer
 */
function azo_elementor_canvas_footer() {
	// quick view markup.
	if ( class_exists( 'woocommerce' ) ) {
		azo_product_action();
	}

	if ( get_theme_mod( 'ajax_search', false ) ) {
		azo_ajax_search_form();
	} else {
		azo_dialog_search_form();
	}

	azo_after_footer();
}

// PARTIAL REFRESH.

if ( ! function_exists( 'azo_menu_slug' ) ) {
	/**
	 * MENU LAYOUT SLUG
	 */
	function azo_menu_slug() {
		/*CUSTOMIZE*/
		$layout = get_query_var( 'menu_layout' ) ? get_query_var( 'menu_layout' ) : 'layout-1';

		// BLOG SINGLE.
		$pid              = get_queried_object_id();
		$single_blog_menu = function_exists( 'fw_get_db_post_option' ) ? fw_get_db_post_option( $pid, 'single_blog_menu' ) : 'default';

		if ( function_exists( 'FW' ) && 'default' !== $single_blog_menu && is_singular( 'post' ) ) :
			$layout = $single_blog_menu;
		endif;

		/*PAGE*/
		$id               = get_queried_object_id();
		$page_menu_layout = true === azo_elementor_page( $id ) ? azo_page_opt( $id, 'page_menu_layout' ) : null;
		if ( isset( $page_menu_layout ) && 'default' !== $page_menu_layout ) {
			$layout = $page_menu_layout;
		}

		return $layout;
	}
}

if ( ! function_exists( 'azo_sticky_header' ) ) {
	/**
	 * STICKY HEADER MENU
	 */
	function azo_sticky_header() {
		?>
		<div class="menu-layout menu-layout-1 menu-layout--classic menu-layout--sticky js-sticky-header">
			<header class="header-box">
				<div class="container" <?php azo_schema_markup( 'navigation' ); ?>>
					<div class="header-container container">
						<div class="header-logo">
							<?php azo_logo_image(); ?>
						</div>

						<div class="theme-menu-box header-menu">
							<span class="screen-reader-text"><?php esc_html_e( 'Primary Menu', 'azo' ); ?></span>
							<?php
							if ( has_nav_menu( 'primary' ) ) :
								wp_nav_menu(
									array(
										'theme_location' => 'primary',
										'menu_class'     => 'theme-primary-menu',
										'container'      => '',
									)
								);
							elseif ( is_user_logged_in() ) :
								?>
								<a class="add-menu" href="<?php echo esc_url( get_admin_url() . 'nav-menus.php' ); ?>"><?php esc_html_e( 'Add Menu', 'azo' ); ?></a>
							<?php endif; ?>
						</div><!-- .theme-menu-box -->

						<div class="header-action">
							<button class="azo-search js-search-button"></button>

							<?php
							if ( class_exists( 'woocommerce' ) ) {
								azo_wc_header_login();

								azo_wc_header_action();
							}
							?>
						</div><!-- .header-action -->
					</div><!-- .header-container -->
				</div><!-- .container -->
			</header><!-- .header-box -->
		</div><!-- .menu-layout-1 -->
		<?php
	}
}

if ( ! function_exists( 'azo_menu_layout' ) ) {
	/**
	 * MENU LAYOUT
	 */
	function azo_menu_layout() {
		/*CUSTOMIZE*/
		$menu_layout = azo_menu_slug();

		switch ( $menu_layout ) {
			case 'layout-1':
				if ( true === get_theme_mod( 'header_1_topbar', false ) ) {
					azo_topbar();
				}
				?>
				<div class="menu-layout menu-layout-1 menu-layout--classic">
					<header class="header-box">
						<div class="header-container">
							<div class="container">
								<button class="menu-toggle-btn"><span></span></button>

								<div class="header-logo">
									<?php azo_logo_image(); ?>
								</div>

								<div class="search-default-azo">
									<?php get_search_form(); ?>
								</div>

								<?php if ( class_exists( 'woocommerce' ) ) { ?>
									<div class="m-col m2-col">
										<div class="header-action">
											<?php
												azo_wc_header_login();

												azo_wc_header_action();
											?>
										</div>
									</div>
								<?php } ?>
							</div>
						</div><!-- .header-container -->

						<div class="navigation-header">
							<div class="container" <?php azo_schema_markup( 'navigation' ); ?>>
								<div class="menu-header">
									<div class="theme-menu-box header-menu">
										<span class="screen-reader-text"><?php esc_html_e( 'Primary Menu', 'azo' ); ?></span>
										<?php
										if ( has_nav_menu( 'primary' ) ) {
											wp_nav_menu(
												array(
													'theme_location' => 'primary',
													'menu_class'     => 'theme-primary-menu',
													'container'      => '',
												)
											);
										} elseif ( is_user_logged_in() ) {
											?>
											<a class="add-menu" href="<?php echo esc_url( get_admin_url() . 'nav-menus.php' ); ?>">
												<?php esc_html_e( 'Add Menu', 'azo' ); ?>
											</a>
										<?php } ?>
									</div><!-- .theme-menu-box -->
								</div>
							</div><!-- .container -->
						</div>
					</header><!-- .header-box -->
					<span id="menu-overlay"></span>
				</div><!-- .menu-layout-1 -->
				<?php
				break;
			case 'layout-2':
				if ( true === get_theme_mod( 'header_2_topbar', false ) ) {
					azo_topbar();
				}
				?>
				<div class="menu-layout menu-layout-2 menu-layout--vertical">
					<header class="header-box">
						<div class="container" <?php azo_schema_markup( 'navigation' ); ?>>
							<div class="header-container">
								<div class="m-col m1-col">
									<button class="menu-toggle-btn"><span></span></button>
								</div>

								<div class="header-logo">
									<?php azo_logo_image(); ?>
								</div><!-- .header-logo -->

								<div class="m-col m2-col">
									<div class="header-action">
										<?php
										if ( class_exists( 'woocommerce' ) ) {
											azo_wc_header_login();

											azo_wc_header_action();
										}
										?>
									</div>
								</div>
							</div><!-- .header-container -->
						</div><!-- .container -->
						<span id="menu-overlay"></span>
					</header><!-- .header-box -->
				</div><!-- .menu-layout-2 -->
				<?php
				break;
			case 'Disable':
				break;
		}
	}
}

if ( ! function_exists( 'azo_page_header_slug' ) ) {
	/**
	 * PAGE HEADER SLUG: RETURN PAGE HEADER LAYOUT VALUE
	 */
	function azo_page_header_slug() {

		$pid = get_queried_object_id();

		/**
		 * CUSTOMIZER
		 */
		$c_header = get_theme_mod( 'c_header_layout', 'layout-1' );

		// WOOCOMMERCE PAGE.
		if ( class_exists( 'woocommerce' ) && is_woocommerce() ) {
			$c_header = get_theme_mod( 'shop_header_layout', 'layout-1' );
		}

		// PAGE.
		$page_header = true === azo_elementor_page( $pid ) ? azo_page_opt( $pid, 'p_header_layout' ) : null;
		if ( isset( $page_header ) && 'default' !== $page_header ) {
			$c_header = $page_header;
		}

		return $c_header;
	}
}


if ( ! function_exists( 'azo_post_related' ) ) {
	/**
	 * Display related post.
	 */
	function azo_post_related() {
		if ( true === get_theme_mod( 'show_navigation', false ) ) {
			$id    = get_queried_object_id();
			$paged = ( get_query_var( 'paged' ) ) ? get_query_var( 'paged' ) : 1;

			$args = array(
				'post_type'           => 'post',
				'post__not_in'        => array( $id ),
				'post_status'         => 'publish',
				'posts_per_page'      => -1,
				'paged'               => $paged,
				'ignore_sticky_posts' => 1,
			);

			$query = new WP_Query( $args );

			if ( $query->have_posts() ) {
				// Previous/next post navigation.
				$next_post     = get_next_post();
				$previous_post = get_previous_post();
				$title_nav     = get_the_title();

				global $post;
				$categories = get_the_category( $post->ID );
				$cat_nav    = $categories[0]->cat_name;

				the_post_navigation(
					array(
						'next_text' => '<span class="azo-article-nav-text">' . $cat_nav . '</span><strong class="title-nav">%title</strong>',
						'prev_text' => '<span class="azo-article-nav-text">' . $cat_nav . '</span><strong class="title-nav">%title</strong>',
					)
				);
				wp_reset_postdata();
			}
		}
	}
}

if ( ! function_exists( 'azo_related_articles' ) ) {
	/**
	 * Azo related articles
	 */
	function azo_related_articles() {
		if ( true === get_theme_mod( 'show_related', false ) ) {
			global $post;
			$tags = wp_get_post_tags( $post->ID );

			if ( $tags ) {
				?>
				<div class="related-articles">
					<h3 class="title-related">
						<?php esc_html_e( 'Related Posts', 'azo' ); ?>
					</h3>

					<div class="related-azo">
						<?php
						$first_tag = $tags[0]->term_id;
						$args      = array(
							'tag__in'             => array( $first_tag ),
							'post__not_in'        => array( $post->ID ),
							'posts_per_page'      => 3,
							'ignore_sticky_posts' => 1,
						);

						$my_query = new WP_Query( $args );

						if ( $my_query->have_posts() ) {
							while ( $my_query->have_posts() ) {
								$my_query->the_post();

								global $post;
								$thumbnail = get_the_post_thumbnail( $post->ID, 'full', array( 'class' => 'full-width' ) );
								?>
									<div class="azo-post-related">
										<?php if ( ! empty( $thumbnail ) ) { ?>
											<div class="box-img">
												<a href="<?php the_permalink(); ?>">
													<?php echo wp_kses_decode_entities( $thumbnail ); ?>
												</a>
											</div>
										<?php } ?>
										<h6 class="title-related-post">
											<a href="<?php the_permalink(); ?>" rel="bookmark" title="Permanent Link to <?php the_title_attribute(); ?>">
												<?php the_title(); ?>
											</a>
										</h6>
										<?php azo_post_info(); ?>
									</div>
								<?php
							}
						}

						wp_reset_postdata();
						?>
					</div>
				</div>
				<?php
			}
		}
	}
}

if ( ! function_exists( 'azo_product_single_sharing' ) ) {
	/**
	 * Azo product single sharing description
	 */
	function azo_product_single_sharing() {
		global $product;
		$id       = $product->get_id();
		$url      = get_permalink( $id );
		$title    = get_the_title( $id );
		$img_id   = $product->get_image_id();
		$img      = wp_get_attachment_image_src( $img_id, 'full' );
		$tags     = get_the_terms( $id, 'product_tag' );
		$tag_list = '';

		if ( $tags && ! is_wp_error( $tags ) ) {
			$tag_list = implode( ', ', wp_list_pluck( $tags, 'name' ) );
		}
		?>
		<span class="theme-social-icon p-shared meta_detail">
			<strong><?php esc_html_e( 'Share On:', 'azo' ); ?></strong>
			<a href="<?php echo esc_url_raw( '//facebook.com/sharer.php?u=' . rawurlencode( $url ) ); ?>" title="<?php echo esc_attr( $title ); ?>" target="_blank" ></a>
			<a href="<?php echo esc_url_raw( '//twitter.com/intent/tweet?url=' . rawurlencode( $url ) . '&text=' . rawurlencode( $title ) . '&hashtags=' . rawurlencode( $tag_list ) ); ?>" title="<?php echo esc_attr( $title ); ?>" target="_blank" ></a>
			<a href="<?php echo esc_url_raw( '//linkedin.com/shareArticle?mini=true&url=' . rawurlencode( $url ) . '&title=' . rawurlencode( $title ) ); ?>" title="<?php echo esc_attr( $title ); ?>" target="_blank" ></a>
			<a href="<?php echo esc_url_raw( '//pinterest.com/pin/create/button/?url=' . rawurlencode( $url ) . '&image_url=' . rawurlencode( $img[0] ) . '&description=' . rawurlencode( $title ) ); ?>" title="<?php echo esc_attr( $title ); ?>" target="_blank" ></a>
			<a href="<?php echo esc_url_raw( '//plus.google.com/share?url=' . rawurlencode( $url ) . '&text=' . rawurlencode( $title ) ); ?>"
				title="<?php echo esc_attr( $title ); ?>" target="_blank" ></a>
		</span>
		<?php
	}
}

if ( ! function_exists( 'azo_get_page_id' ) ) {
	/**
	 * Get page id
	 *
	 * @return int $page_id Page id
	 */
	function azo_get_page_id() {
		$page_id = get_queried_object_id();

		if ( class_exists( 'woocommerce' ) ) {
			if ( is_shop() ) {
				$page_id = get_option( 'woocommerce_shop_page_id' );
			} elseif ( is_product_category() ) {
				$page_id = false;
			}
		}

		return $page_id;
	}
}

if ( ! function_exists( 'azo_is_woocommerce_activated' ) ) {
	/**
	 * Query WooCommerce activation
	 */
	function azo_is_woocommerce_activated() {
		return class_exists( 'woocommerce' ) ? true : false;
	}
}

if ( ! function_exists( 'azo_is_elementor_activated' ) ) {
	/**
	 * Check Elementor active
	 *
	 * @return     bool
	 */
	function azo_is_elementor_activated() {
		return defined( 'ELEMENTOR_VERSION' );
	}
}

if ( ! function_exists( 'azo_is_elementor_page' ) ) {
	/**
	 * Detect Elementor Page editor with current page
	 *
	 * @param int $page_id The page id.
	 * @return     bool
	 */
	function azo_is_elementor_page( $page_id = false ) {
		if ( ! azo_is_elementor_activated() ) {
			return false;
		}

		if ( ! $page_id ) {
			$page_id = azo_get_page_id();
		}

		$edit_mode = get_post_meta( $page_id, '_elementor_edit_mode', true );
		$edit_mode = 'builder' === $edit_mode ? true : false;

		if ( ! $page_id ) {
			$edit_mode = false;
		}

		return $edit_mode;
	}
}

/**
 * Change the breadcrumb separator
 *
 * @param  [type] $defaults type.
 */
function azo_wcc_change_breadcrumb_delimiter( $defaults ) {
	// Change the breadcrumb delimeter from '/' to '>'.
	return array(
		'delimiter'   => ' <span class="ion-ios-arrow-right"></span> ',
		'wrap_before' => '<nav class="woocommerce-breadcrumb">',
		'wrap_after'  => '</nav>',
		'before'      => '',
		'after'       => '',
		'home'        => _x( 'Home', 'breadcrumb', 'azo' ),
	);
}
add_filter( 'woocommerce_breadcrumb_defaults', 'azo_wcc_change_breadcrumb_delimiter' );

if ( ! function_exists( 'azo_page_header' ) ) {
	/**
	 * PAGE HEADER
	 */
	function azo_page_header() {
		if ( is_404() ) {
			return;
		}

		global $post;
		if ( is_page() ) {
			$bg_header = get_the_post_thumbnail_url( $post->ID, 'full' );
		}

		$bf_page_header = '';
		if ( class_exists( 'acf' ) ) {
			$bf_page_header = get_field( 'bf_page_header' );
		}

		if ( false === $bf_page_header ) {
			return;
		}

		$page_header_img = '';
		if ( is_page() && false !== $bg_header ) {
			$page_header_img = 'on-background';
		}

		if ( class_exists( 'woocommerce' ) ) {
			if ( is_shop() ) {
				$page_header_img = 'on-background';
			}
		}
		?>
		<div id="theme-page-header">
			<div class="page-header <?php echo esc_attr( $page_header_img ); ?>"
				<?php if ( has_post_thumbnail() && is_page() ) { ?>
					style="background-image: url('<?php echo esc_url( $bg_header ); ?>')"
				<?php } ?>>
				<div class="container">
					<div id="theme-bread">
						<?php
						if ( class_exists( 'woocommerce' ) ) {
							if ( ! is_product() ) {
								azo_page_title();
							}
						}

						/*BREADCRUMBS*/
						if ( class_exists( 'woocommerce' ) ) {
							if ( false !== get_theme_mod( 'show_breadcrumd', true ) ) {
								if ( false !== get_theme_mod( 'show_breadcrumd_post', true ) ) {
									woocommerce_breadcrumb();
								} elseif ( false !== get_theme_mod( 'show_breadcrumd_shop_single', true ) ) {
									woocommerce_breadcrumb();
								}
							}
						}

						if ( defined( 'WC_PLUGIN_FILE' ) ) {
							azo_prev_next_product();
						}
						?>
					</div>
				</div>
			</div>
		</div>
		<?php
	}
}

if ( ! function_exists( 'azo_page_title' ) ) {
	/**
	 * PAGE TITLE
	 */
	function azo_page_title() {
		if ( false === get_theme_mod( 'show_title_page', true ) ) {
			return;
		}

		/*PAGE TITLE*/
		$title = get_the_title();

		/*BLOG TITLE*/
		$blog_title = get_theme_mod( 'blog_title', 'Blog' );

		/*SHOP TITLE*/
		$shop_title = get_theme_mod( 'shop_title', 'Shop Product' );

		?>
		<h1 class="page-title entry-title">
			<?php
			if ( is_day() ) :
				printf( esc_html__( 'Daily Archives: %s', 'azo' ), get_the_date() );
			elseif ( is_month() ) :
				printf( esc_html__( 'Monthly Archives: %s', 'azo' ), get_the_date( esc_html_x( 'F Y', 'monthly archives date format', 'azo' ) ) );
			elseif ( is_home() ) :
				echo esc_html( $blog_title );
			elseif ( is_author() ) :
				$author = ( get_query_var( 'author_name' ) ) ? get_user_by( 'slug', get_query_var( 'author_name' ) ) : get_userdata( get_query_var( 'author' ) );
				echo esc_html( $author->display_name );
			elseif ( is_year() ) :
				printf( esc_html__( 'Yearly Archives: %s', 'azo' ), get_the_date( esc_html_x( 'Y', 'yearly archives date format', 'azo' ) ) );
			elseif ( class_exists( 'woocommerce' ) && ( is_product_tag() || is_tag() ) ) :
				esc_html_e( 'Tags: ', 'azo' );
				single_tag_title();
			elseif ( class_exists( 'woocommerce' ) && is_product() ) :
				echo esc_html( $title );
			elseif ( is_page() ) :
				echo ! empty( $title ) ? esc_html( $title ) : esc_html__( 'This post has no title', 'azo' );
			elseif ( is_single() && 'post' !== get_post_type() ) :
				echo esc_html( get_the_title() );
			elseif ( is_single() && 'post' === get_post_type() ) :
				echo esc_html( get_the_title() );
			elseif ( is_tax() ) :
				global $wp_query;
				$term      = $wp_query->get_queried_object();
				$tex_title = $term->name;
				echo esc_html( $tex_title );
			elseif ( is_search() ) :
				global $wp_query;
				echo esc_html( $wp_query->found_posts );
				echo esc_html_e( ' search results found for "', 'azo' );
				the_search_query();
				echo esc_html( '"' );
			elseif ( class_exists( 'woocommerce' ) && is_shop() ) :
				echo esc_html( $shop_title );
			elseif ( is_archive() ) :
				esc_html_e( 'Archives', 'azo' );
			else :
				global $wp_query;
				$term      = $wp_query->get_queried_object();
				$tex_title = $term->name;
				echo esc_html( $tex_title );
			endif;
			?>
		</h1>
		<?php
	}
}

if ( ! function_exists( 'azo_footer_display' ) ) {
	/**
	 * FOOTER SLUG
	 */
	function azo_footer_display() {
		$pid = get_queried_object_id();

		/*CUSTOMIZE*/
		$show_footer = get_theme_mod( 'show_footer', true );

		/* BLOG SINGLE */
		$single_blog_footer = function_exists( 'fw_get_db_post_option' ) ? fw_get_db_post_option( $pid, 'p_footer' ) : 'default';

		if ( function_exists( 'FW' ) && 'default' !== $single_blog_footer && is_singular( 'post' ) ) :
			$show_footer = 'enable' === $single_blog_footer ? true : false;
		endif;

		/*PAGE*/
		$id                 = get_queried_object_id();
		$page_footer_layout = true === azo_elementor_page( $id ) ? azo_page_opt( $id, 'p_footer_layout' ) : null;
		if ( isset( $page_footer_layout ) && 'default' !== $page_footer_layout ) {
			$show_footer = 'enable' === $page_footer_layout ? true : false;
		}

		return $show_footer;
	}
}

if ( ! function_exists( 'azo_default_footer_copyright' ) ) {
	/**
	 * Footer Copyright
	 */
	function azo_default_footer_copyright() {
		return sprintf(
			/* Translators: 1: Years, 2: Name, 3: Author */
			esc_html__( '&copy; %1$s %2$s. All Rights Reserved. Designed by %3$s', 'azo' ),
			gmdate( 'Y' ),
			get_bloginfo( 'name' ),
			'<a href="https://themeforest.net/user/boostifythemes" rel="nofollow">BoostifyThemes</a>.'
		);
	}
}

if ( ! function_exists( 'azo_footer' ) ) {
	/**
	 * FOOTER
	 */
	function azo_footer() {
		if ( ! azo_footer_display() ) {
			return;
		}

		$column          = get_theme_mod( 'ft_column', 4 );
		$copyright       = get_theme_mod( 'ft_copyright', '' );
		$copyright       = ! empty( $copyright ) ? $copyright : azo_default_footer_copyright();
		$right_bot_right = get_theme_mod( 'ft_bot_right' );

		/*WIDGET*/
		if ( is_active_sidebar( 'footer-widget' ) ) {
			?>
			<div class="footer-top">
				<div class="container">
					<div class="row widget-box footer-col-<?php echo esc_attr( $column ); ?>">
						<?php dynamic_sidebar( 'footer-widget' ); ?>
					</div>
				</div>
			</div>
		<?php } ?>

		<?php /*BASE*/ ?>
		<div class="footer-bot">
			<div class="container">
				<div class="footer-left">
					<div class="footer-logo">
						<?php azo_footer_logo(); ?>
					</div>
				</div>

				<div class="footer-center">
					<?php echo do_shortcode( $copyright ); ?>
				</div>

				<div class="footer-bot-right">
					<?php echo do_shortcode( $right_bot_right ); ?>
				</div>
			</div>
		</div>
		<?php
	}
}

if ( ! function_exists( 'azo_footer_logo' ) ) {
	/**
	 * FOOTER LOGO
	 */
	function azo_footer_logo() {
		$footer_logo = get_theme_mod( 'ft_logo' );

		if ( false === $footer_logo ) {
			echo '';
		} else {
			?>
			<a href="<?php echo esc_url( home_url( '/' ) ); ?>">
				<img src="<?php echo esc_url( get_theme_mod( 'ft_logo' ) ); ?>" alt="<?php esc_attr_e( 'Footer logo', 'azo' ); ?>">
			</a>
			<?php
		}
	}
}

if ( ! function_exists( 'azo_topbar' ) ) {
	/**
	 * FOOTER LOGO
	 */
	function azo_topbar() {
		$topbar_left_content = get_theme_mod( 'topbar_left', '+01 532 485 283<span class="space-top">sale@azo.com</span>' );

		$topbar_center_content = get_theme_mod( 'topbar_center', 'Free shipping on all orders over $75.00' );

		$topbar_right_content = get_theme_mod( 'topbar_right', '<span class="gray">Language:</span><div class="dropdown"><span class="dropdown__current">English <i class="fa fa-angle-down"></i></span><div class="dropdown__content"><a href="#">English</a><a href="#">French</a></div></div><span class="gray">Currency:</span><div class="dropdown"><span class="dropdown__current">USD <i class="fa fa-angle-down"></i></span><div class="dropdown__content"><a href="#">USD</a><a href="#">EUR</a></div></div>' );
		?>
		<div class="topbar">
			<div class="container">
				<div class="topbar__container">
					<div class="topbar__left">
						<?php
						if ( ! empty( $topbar_left_content ) ) {
							echo do_shortcode( $topbar_left_content );
						}
						?>
					</div><!-- .topbar__left -->

					<div class="topbar__center">
						<?php
						if ( ! empty( $topbar_center_content ) ) {
							echo do_shortcode( $topbar_center_content );
						}
						?>
					</div>

					<div class="topbar__right">
						<?php
						if ( ! empty( $topbar_right_content ) ) {
							echo do_shortcode( $topbar_right_content );
						}
						?>
					</div><!-- .topbar__right -->
				</div><!-- .topbar__container -->
			</div><!-- .container -->
		</div><!-- .topbar -->
		<?php
	}
}

// AJAX.

if ( class_exists( 'woocommerce' ) ) {
	add_action( 'wp_enqueue_scripts', 'azo_ajax_url', 999 );
	/**
	 * CREATE AJAX URL
	 */
	function azo_ajax_url() {
		wp_localize_script(
			'azo-custom',
			'azo_ajax',
			array(
				'url'      => admin_url( 'admin-ajax.php' ),
				'nonce'    => wp_create_nonce( 'azo_product_nonce' ),
				'cart_url' => wc_get_cart_url(),
			)
		);
	}
}

if ( ! function_exists( 'azo_wc_sidebar_action' ) ) {
	/**
	 * ICON HEADER EMENU
	 */
	function azo_wc_sidebar_action() {
		if ( ! class_exists( 'woocommerce' ) ) {
			return;
		}

		global $woocommerce;
		$page_account = get_option( 'woocommerce_myaccount_page_id' );
		$page_logout  = wp_logout_url( get_permalink( $page_account ) );

		if ( 'yes' === get_option( 'woocommerce_force_ssl_checkout' ) ) {
			$logout_url = str_replace( 'https:', 'https:', $logout_url );
		}

		$count = $woocommerce->cart->cart_contents_count;
		?>
		<ul class="sidebar-actions custom-sidebar-actions">
			<li class="sidebar-action custom-sidebar-login">
				<?php if ( ! is_user_logged_in() ) : ?>
					<a href="<?php echo esc_url( get_permalink( $page_account ) ); ?>" class="sidebar-action-link">
						<span class="azo-icon-user sidebar-action-icon"></span> <span class="sidebar-action-text">
						<?php esc_html_e( 'Login', 'azo' ); ?>
					</span>
					</a>
				<?php else : ?>
					<a href="<?php echo esc_url( get_permalink( $page_account ) ); ?>" class="sidebar-action-link">
						<span class="azo-icon-user sidebar-action-icon"></span>
						<span class="sidebar-action-text"><?php esc_html_e( 'Dashboard', 'azo' ); ?></span>
					</a>
				<?php endif; ?>
			</li>
			<li class="sidebar-action custom-sidebar-cart">
				<a href="<?php echo esc_url( wc_get_cart_url() ); ?>" class="sidebar-action-link js-cart-button">
					<span class="azo-icon-cart sidebar-action-icon"></span>
					<span class="sidebar-action-text"><?php esc_html_e( 'Your Cart', 'azo' ); ?></span>
					<span class="sidebar-action-cart shop-cart-count"><?php echo esc_html( $count ); ?></span>
				</a>
			</li>
			<?php if ( is_user_logged_in() ) : ?>
				<li class="sidebar-action custom-sidebar-cart">
					<a href="<?php echo esc_url( $page_logout ); ?>" class="sidebar-action-link">
						<span class="azo-icon-log-out sidebar-action-icon"></span>
						<span class="sidebar-action-text"><?php esc_html_e( 'Logout', 'azo' ); ?></span>
					</a>
				</li>
			<?php endif; ?>
		</ul>
		<?php
	}
}

function render_button( $args ) {
	$defaults = [
		'text'          => '',
		'link'          => [
			'url'         => '',
			'is_external' => false,
			'nofollow'    => false,
		],
		'style'         => 'flat',
		'size'          => 'nm',
		'icon'          => '',
		'icon_align'    => 'left',
		'extra_class'   => '',
		'class'         => 'tm-button',
		'id'            => '',
		'wrapper_class' => '',
	];

	$args = wp_parse_args( $args, $defaults );
	extract( $args );

	$button_attrs = array();

	$button_classes   = array( $class );
	$button_classes[] = 'style-' . $style;
	$button_classes[] = 'tm-button-' . $size;

	if ( ! empty( $extra_class ) ) {
		$button_classes[] = $extra_class;
	}

	$button_attrs['class'] = implode( ' ', $button_classes );

	if ( ! empty( $id ) ) {
		$button_attrs['id'] = $id;
	}

	$button_tag = 'div';

	if ( ! empty( $link['url'] ) ) {
		$button_tag = 'a';

		$button_attrs['href'] = $link['url'];

		if ( ! empty( $link['is_external'] ) ) {
			$button_attrs['target'] = $link['_blank'];
		}

		if ( ! empty( $link['nofollow'] ) ) {
			$button_attrs['rel'] = $link['nofollow'];
		}
	}

	$attributes_str = '';

	if ( ! empty( $button_attrs ) ) {
		foreach ( $button_attrs as $attribute => $value ) {
			$attributes_str .= ' ' . $attribute . '="' . esc_attr( $value ) . '"';
		}
	}

	$wrapper_classes = 'tm-button-wrapper';
	if ( ! empty( $wrapper_class ) ) {
		$wrapper_classes .= " $wrapper_class";
	}
	?>
	<div class="<?php echo esc_attr( $wrapper_classes ); ?>">
		<?php printf( '<%1$s %2$s>', $button_tag, $attributes_str ); ?>
			<div class="button-content-wrapper">
				<?php if ( ! empty( $text ) ) : ?>
					<span class="button-text"><?php echo esc_html( $text ); ?></span>
				<?php endif; ?>

				<?php if ( 'bottom-line-winding' === $style ) : ?>
					<span class="line-winding">
						<svg width="42" height="6" viewBox="0 0 42 6" fill="none" xmlns="https://www.w3.org/2000/svg"> <path fill-rule="evenodd" clip-rule="evenodd" d="M0.29067 2.60873C1.30745 1.43136 2.72825 0.72982 4.24924 0.700808C5.77022 0.671796 7.21674 1.31864 8.27768 2.45638C8.97697 3.20628 9.88872 3.59378 10.8053 3.5763C11.7218 3.55882 12.6181 3.13683 13.2883 2.36081C14.3051 1.18344 15.7259 0.481897 17.2469 0.452885C18.7679 0.423873 20.2144 1.07072 21.2753 2.20846C21.9746 2.95836 22.8864 3.34586 23.8029 3.32838C24.7182 3.31092 25.6133 2.89009 26.2831 2.11613C26.2841 2.11505 26.285 2.11396 26.2859 2.11288C27.3027 0.935512 28.7235 0.233974 30.2445 0.204962C31.7655 0.17595 33.212 0.822796 34.2729 1.96053C34.9722 2.71044 35.884 3.09794 36.8005 3.08045C37.7171 3.06297 38.6134 2.64098 39.2836 1.86496C39.6445 1.44697 40.276 1.40075 40.694 1.76173C41.112 2.12271 41.1582 2.75418 40.7972 3.17217C39.7804 4.34954 38.3597 5.05108 36.8387 5.08009C35.3177 5.1091 33.8712 4.46226 32.8102 3.32452C32.1109 2.57462 31.1992 2.18712 30.2826 2.2046C29.3674 2.22206 28.4723 2.64289 27.8024 3.41684C27.8015 3.41793 27.8005 3.41901 27.7996 3.42009C26.7828 4.59746 25.362 5.299 23.841 5.32801C22.3201 5.35703 20.8735 4.71018 19.8126 3.57244C19.1133 2.82254 18.2016 2.43504 17.285 2.45252C16.3685 2.47 15.4722 2.89199 14.802 3.66802C13.7852 4.84539 12.3644 5.54693 10.8434 5.57594C9.32242 5.60495 7.8759 4.9581 6.81496 3.82037C6.11568 3.07046 5.20392 2.68296 4.28738 2.70044C3.37083 2.71793 2.47452 3.13992 1.80434 3.91594C1.44336 4.33393 0.811887 4.38015 0.393899 4.01917C-0.0240897 3.65819 -0.0703068 3.02672 0.29067 2.60873Z" fill="#E8C8B3"/>
						</svg>
					</span>
				<?php endif; ?>
			</div>
		<?php printf( '</%1$s>', $button_tag ); ?>
	</div>
	<?php
}

/**
 * Render rating
 *
 * @param  integer $rating type.
 * @param  array   $args   type.
 */
function render_rating( $rating = 5, $args = array() ) {
	$default = array(
		'style'         => '01',
		'wrapper_class' => '',
		'echo'          => true,
	);

	$args = wp_parse_args( $args, $default );

	$el_classes = 'tm-star-rating style-' . $args['style'];
	if ( ! empty( $args['wrapper_class'] ) ) {
		$el_classes .= " {$args['wrapper_class']}";
	}

	$full_stars = intval( $rating );

	$template  = '';
	$template .= str_repeat( '<span class="tm-star-full"></span>', $full_stars );

	$half_star = floatval( $rating ) - $full_stars;

	if ( 0 !== $half_star ) {
		$template .= '<span class="tm-star-half"></span>';
	}

	$empty_stars = intval( 5 - $rating );
	$template   .= str_repeat( '<span class="tm-star-empty"></span>', $empty_stars );

	$template = '<div class="' . esc_attr( $el_classes ) . '">' . $template . '</div>';

	if ( $args['echo'] ) {
		echo '' . $template;
	}

	return $template;
}

/**
 * Azo lost password login popup
 */
function azo_add_lost_password_link() {
	return '<a class="lost-password" href="' . esc_url( wp_lostpassword_url() ) . '">Lost password?</a>';
}
add_action( 'login_form_middle', 'azo_add_lost_password_link' );

if ( ! function_exists( 'azo_after_footer' ) ) {
	/**
	 * After footer
	 */
	function azo_after_footer() {
		?>
		<a href="#" class="scroll-to-top js-to-top">
			<i class="ion-chevron-up"></i>
		</a>
		<?php
	}
}

// INLINE CSS.

/**
 * [azo_inline_style description]
 */
function azo_inline_style() {

	$css = '';

	// Mega menu background.
	// Get primary menu-id.
	$menu_ids        = get_nav_menu_locations();
	$bg_image_output = '';
	if ( $menu_ids ) {
		$css .= '@media ( min-width: 992px ) {';
		foreach ( $menu_ids as $k ) {
			$menu_id = wp_get_nav_menu_items( $k );
			if ( $menu_id ) {
				foreach ( $menu_id as $v ) {
					$options = function_exists( 'fw_ext_mega_menu_get_db_item_option' ) ? fw_ext_mega_menu_get_db_item_option( $v->ID, 'row' ) : '';
					if ( isset( $options['menu_bg_image'] ) && ! empty( $options['menu_bg_image'] ) ) {
						$bg_image_output .= 'background-image: url(' . esc_url( $options['menu_bg_image']['url'] ) . ');';
					}
					if ( isset( $options['menu_bg_position'] ) && $options['menu_bg_position'] ) {
						$bg_image_output .= 'background-position: ' . str_replace( '-', ' ', $options['menu_bg_position'] ) . ';';
					}
					if ( isset( $options['menu_bg_repeat'] ) && $options['menu_bg_repeat'] ) {
						$bg_image_output .= 'background-repeat: ' . $options['menu_bg_repeat'] . ';';
					}
					if ( isset( $options['menu_bg_size'] ) && $options['menu_bg_size'] ) {
						$bg_image_output .= 'background-size: ' . $options['menu_bg_size'] . ';';
					}

					if ( $bg_image_output && isset( $options['menu_bg_image'] ) && ! empty( $options['menu_bg_image'] ) ) {
						$css .= '.theme-primary-menu:not(.theme-sidebar-menu) .menu-item-' . $v->ID . ' .mega-menu-row{' . $bg_image_output . '}';
					}
				}
			}
		}
		$css .= '}';
	}

	wp_add_inline_style( 'azo-theme-style', $css );
}
add_action( 'wp_enqueue_scripts', 'azo_inline_style', 98 );

/**
 * [azo_import_files description]
 */
function azo_import_files() {
	return array(
		array(
			'import_file_name'             => 'Demo Full',
			'local_import_file'            => trailingslashit( AZO_DIR ) . 'demo/azo-content.xml',
			'local_import_widget_file'     => trailingslashit( AZO_DIR ) . 'demo/azo-widget.wie',
			'local_import_customizer_file' => trailingslashit( AZO_DIR ) . 'demo/azo-customize.dat',
			'import_notice'                => esc_html__( 'Please actived the required plugins before installing the demo. After you import this demo, you will have to setup the slider separately.', 'azo' ),
		),
	);
}
add_filter( 'pt-ocdi/import_files', 'azo_import_files' );

/**
 * FUNCTION SETTING AFTER IMPORT DEMO
 *
 * @param  [type] $selected_import type.
 */
function azo_after_import( $selected_import ) {

	if ( 'Demo Full' === $selected_import['import_file_name'] ) {
		// Set Menu.
		$primary = get_term_by( 'name', 'Menu 1', 'nav_menu' );
		set_theme_mod(
			'nav_menu_locations',
			array(
				'primary' => $primary->term_id,
			)
		);

		update_option( 'posts_per_page', 5 );
		update_option( 'permalink_structure', '/%postname%/' );
		// Set Front page.
		$font_page = get_page_by_title( 'Home 1' );
		$post_page = get_page_by_title( 'blog' );
		if ( isset( $font_page->ID ) ) {
			update_option( 'page_on_front', $font_page->ID );
			update_option( 'show_on_front', 'page' );
		}

		// Set post page.
		if ( isset( $post_page->ID ) ) {
			update_option( 'page_for_posts', $post_page->ID );
			update_option( 'show_on_front', 'page' );
		}

		// Import Revolution Slider.
		if ( class_exists( 'RevSlider' ) ) {
			$slider_array = array(
				AZO_DIR . 'demo/slider-home-1.zip',
				AZO_DIR . 'demo/slider-home-2.zip',
				AZO_DIR . 'demo/slider-home-4.zip',
				AZO_DIR . 'demo/slider-home-5.zip',
				AZO_DIR . 'demo/slider-home-6.zip',
				AZO_DIR . 'demo/slider-home-6-1.zip',
			);

			$slider = new RevSlider();

			foreach ( $slider_array as $filepath ) {
				$slider->importSliderFromPost( true, true, $filepath );
			}
		}
	}
}
add_action( 'pt-ocdi/after_import', 'azo_after_import' );
