<?php
/**
 * Azo Team Member
 *
 * @package azo
 */

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use Elementor\Widget_Base;
use Elementor\Controls_Manager;

/**
 * Azo Team Member
 */
class Azo_Team_Member extends Widget_Base {
	/**
	 * Get widget name.
	 *
	 * Retrieve team member widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'azo-team-member';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve team member widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Team Member', 'azo' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve team member widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-user-circle-o';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the icon widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return array( 'azo-theme' );
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return array( 'azo-team-member' );
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 */
	protected function _register_controls() { // phpcs:ignore
		$this->start_controls_section(
			'section_team_member',
			array(
				'label' => esc_html__( 'Team Member', 'azo' ),
			)
		);

		// Slides to show.
		$this->add_responsive_control(
			'slides_to_show',
			array(
				'label'                 => esc_html__( 'Slides to show', 'azo' ),
				'type'                  => Controls_Manager::NUMBER,
				'min'                   => 1,
				'max'                   => 5,
				'step'                  => 1,
				'devices'               => array( 'desktop', 'tablet', 'mobile' ),
				'slides_to_show_mobile' => array(
					'default' => 1,
				),
				'slides_to_show_tablet' => array(
					'default' => 2,
				),
				'slides_to_show'        => array(
					'default' => 2,
				),
			)
		);

		$repeater = new \Elementor\Repeater();

		$repeater->add_control(
			'image',
			array(
				'label'   => esc_html__( 'Choose Image', 'azo' ),
				'type'    => Controls_Manager::MEDIA,
				'dynamic' => array(
					'active' => true,
				),
				'default' => array(
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				),
			)
		);

		$repeater->add_control(
			'name',
			array(
				'label'       => esc_html__( 'Name', 'azo' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => esc_html__( 'Garry Devin', 'azo' ),
				'placeholder' => esc_attr__( 'Member Name', 'azo' ),
				'label_block' => true,
			)
		);

		$repeater->add_control(
			'description',
			array(
				'label'       => esc_html__( 'Description', 'azo' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => esc_html__( 'Founder / Ceo', 'azo' ),
				'placeholder' => esc_attr__( 'Description', 'azo' ),
				'label_block' => true,
			)
		);

		$repeater->add_control(
			'member_contact_1',
			array(
				'label'            => esc_html__( 'Icon 1', 'azo' ),
				'type'             => Controls_Manager::ICON,
				'label_block'      => true,
				'fa4compatibility' => 'icon',
				'default'          => 'fa fa-facebook',
			)
		);

		$repeater->add_control(
			'link_1',
			array(
				'label'       => esc_html__( 'Link 1', 'azo' ),
				'type'        => Controls_Manager::URL,
				'label_block' => true,
				'default'     => array(
					'is_external' => 'true',
				),
				'placeholder' => esc_attr( 'https://member-link.com' ),
			)
		);

		$repeater->add_control(
			'member_contact_2',
			array(
				'label'            => esc_html__( 'Icon 2', 'azo' ),
				'type'             => Controls_Manager::ICON,
				'label_block'      => true,
				'fa4compatibility' => 'icon',
				'default'          => 'fa fa-twitter',
			)
		);

		$repeater->add_control(
			'link_2',
			array(
				'label'       => esc_html__( 'Link 2', 'azo' ),
				'type'        => Controls_Manager::URL,
				'label_block' => true,
				'default'     => array(
					'is_external' => 'true',
				),
				'placeholder' => esc_attr( 'https://member-link.com' ),
			)
		);

		$repeater->add_control(
			'member_contact_3',
			array(
				'label'            => esc_html__( 'Icon 3', 'azo' ),
				'type'             => Controls_Manager::ICON,
				'label_block'      => true,
				'fa4compatibility' => 'icon',
				'default'          => 'fa fa-google-plus',
			)
		);

		$repeater->add_control(
			'link_3',
			array(
				'label'       => esc_html__( 'Link 3', 'azo' ),
				'type'        => Controls_Manager::URL,
				'label_block' => true,
				'default'     => array(
					'is_external' => 'true',
				),
				'placeholder' => esc_attr( 'https://member-link.com' ),
			)
		);

		$repeater->add_control(
			'member_contact_4',
			array(
				'label'            => esc_html__( 'Icon 4', 'azo' ),
				'type'             => Controls_Manager::ICON,
				'label_block'      => true,
				'fa4compatibility' => 'icon',
				'default'          => 'fa fa-pinterest-p',
			)
		);

		$repeater->add_control(
			'link_4',
			array(
				'label'       => esc_html__( 'Link 4', 'azo' ),
				'type'        => Controls_Manager::URL,
				'label_block' => true,
				'default'     => array(
					'is_external' => 'true',
				),
				'placeholder' => esc_attr( 'https://member-link.com' ),
			)
		);

		$this->add_control(
			'teammember',
			array(
				'name'        => esc_html__( 'Teammber name', 'azo' ),
				'type'        => Controls_Manager::REPEATER,
				'fields'      => $repeater->get_controls(),
				'default'     => array(
					array(),
					array(),
					array(),
					array(),
				),
				'title_field' => '{{{ name }}}',
			)
		);

		// autoplay.
		$this->add_control(
			'autoplay',
			array(
				'label'        => esc_html__( 'On/Off autoplay', 'azo' ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'On', 'azo' ),
				'label_off'    => esc_html__( 'Off', 'azo' ),
				'return_value' => 'yes',
				'default'      => 'yes',
			)
		);

		$this->add_control(
			'dots',
			array(
				'label'        => esc_html__( 'Show Dots', 'azo' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'no',
				'label_on'     => esc_html__( 'No', 'azo' ),
				'label_off'    => esc_html__( 'Yes', 'azo' ),
				'return_value' => 'yes',
			)
		);

		$this->add_control(
			'arrow',
			array(
				'label'        => esc_html__( 'Show Arrow', 'azo' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'no',
				'label_on'     => esc_html__( 'No', 'azo' ),
				'label_off'    => esc_html__( 'Yes', 'azo' ),
				'return_value' => 'yes',
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style',
			array(
				'label' => esc_html__( 'Avatar', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'hover_heading',
			array(
				'label' => esc_html__( 'Hover', 'azo' ),
				'type'  => Controls_Manager::HEADING,
			)
		);

		$this->add_control(
			'hover_animation',
			array(
				'label' => esc_html__( 'Hover Animation', 'azo' ),
				'type'  => Controls_Manager::HOVER_ANIMATION,
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style-item',
			array(
				'label' => esc_html__( 'Padding Item', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_responsive_control(
			'box_padding',
			array(
				'label'     => __( 'Padding', 'azo' ),
				'type'      => Controls_Manager::DIMENSIONS,
				'selectors' => array(
					'{{WRAPPER}} .azo-member-contact-info' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'name_style',
			array(
				'label' => esc_html__( 'Member Name', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'name_color',
			array(
				'label'     => esc_html__( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#2f2f30',
				'selectors' => array(
					'{{WRAPPER}} .azo-member-name .name' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			array(
				'name'     => 'name_typography',
				'selector' => '{{WRAPPER}} .azo-member-name .name',
			)
		);

		$this->add_control(
			'name_space',
			array(
				'label'     => esc_html__( 'Space', 'azo' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min' => 0,
						'max' => 100,
					),
				),
				'default'   => array(
					'size' => 0,
				),
				'selectors' => array(
					'{{WRAPPER}} .azo-member-name .name' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'description_style',
			array(
				'label' => esc_html__( 'Member Description', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);
		$this->add_control(
			'description_color',
			array(
				'label'     => esc_html__( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#959595',
				'selectors' => array(
					'{{WRAPPER}} .azo-member-description' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			array(
				'name'     => 'description_typography',
				'selector' => '{{WRAPPER}} .azo-member-description',
			)
		);
		$this->end_controls_section();

		$this->start_controls_section(
			'style_icon_contact',
			array(
				'label' => esc_html__( 'Icon Contact', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_responsive_control(
			'icon_typo',
			array(
				'label'     => __( 'Icon Typography', 'azo' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min' => 1,
						'max' => 300,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .azo-contact-icon li' => 'font-size: {{SIZE}}{{UNIT}};',
				),
				'separator' => 'before',
			)
		);

		$this->add_control(
			'icon_primary_color',
			array(
				'label'     => esc_html__( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#9d9d9d',
				'selectors' => array(
					'{{WRAPPER}} .azo-contact-icon li a' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'icon_primary_color_hover',
			array(
				'label'     => esc_html__( 'Color Hover', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#9d9d9d',
				'selectors' => array(
					'{{WRAPPER}} .azo-contact-icon li a:hover' => 'color: {{VALUE}};',
				),
			)
		);

		$this->add_responsive_control(
			'icon_space',
			array(
				'label'              => esc_html__( 'Space Icon', 'azo' ),
				'type'               => Controls_Manager::DIMENSIONS,
				'size_units'         => array( 'px', '%', 'em' ),
				'allowed_dimensions' => 'horizontal',
				'placeholder'        => array(
					'top'    => 'auto',
					'right'  => '',
					'bottom' => 'auto',
					'left'   => '',
				),
				'selectors'          => array(
					'{{WRAPPER}} .azo-contact-icon li' => 'margin-left: {{LEFT}}{{UNIT}}; margin-right: {{RIGHT}}{{UNIT}};',
				),
			)
		);
		$this->end_controls_section();

		$this->start_controls_section(
			'style-dots',
			array(
				'label' => esc_html__( 'Dots', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'color-dots',
			array(
				'label'     => esc_html__( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .teammember-slider-dots .dots-bullet:after' => 'background-color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'color-dots_active',
			array(
				'label'     => esc_html__( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .teammember-slider-dots .slick-active .dots-bullet:after' => 'background-color: {{VALUE}};',
				),
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 */
	protected function render() {
		$settings     = $this->get_settings_for_display();
		$allowed_html = azo_shape_space_allowed_html();
		$play         = 'yes' === $settings['autoplay'] ? 1 : 0;
		$dot          = $settings['dots'];
		$arrow        = $settings['arrow'];

		$teammember = $settings['teammember'];

		?>
		<div class="azo-teammember-slider-widget">
			<div class="teammember-slider-wrapper" data-show="<?php echo esc_attr( $settings['slides_to_show'] ); ?>" data-show-tablet="<?php echo esc_attr( $settings['slides_to_show_tablet'] ); ?>" data-show-mobile="<?php echo esc_attr( $settings['slides_to_show_mobile'] ); ?>" data-autoplay="<?php echo esc_attr( $play ); ?>">

				<?php if ( 'yes' === $arrow ) { ?>
					<div class="arrows-widget">
						<button class="azo-arrow arrows-prev"></button>
						<button class="azo-arrow arrows-next"></button>
					</div>
				<?php } ?>

				<div class="js-teammember-slider teammember-slider-list" data-dot="<?php echo esc_attr( $dot ); ?>">
					<?php foreach ( $teammember as $team ) { ?>
						<div class="azo-team-member">
							<div class="azo-member-avatar">
								<div class="avatar <?php echo 'elementor-animation-' . esc_attr( $settings['hover_animation'] ); ?>">
									<img src="<?php echo esc_url( $team['image']['url'] ); ?>" alt="<?php esc_attr_e( 'teammember-image', 'azo' ); ?>">
								</div>
							</div>
							<div class="azo-member-contact-info">
								<div class="azo-member-name">
									<h2 class="name"><?php echo esc_html( $team['name'] ); ?></h2>
								</div>
								<div class="azo-member-content-bottom">
									<p class="azo-member-description">
										<?php echo esc_html( $team['description'] ); ?>
									</p>

									<ul class="azo-contact-icon azo-member-list-social">
										<li>
											<a href="<?php echo esc_url( $team['link_1']['url'] ); ?>" class="<?php echo esc_attr( $team['member_contact_1'] ); ?>"></a>
										</li>
										<li>
											<a href="<?php echo esc_url( $team['link_2']['url'] ); ?>" class="<?php echo esc_attr( $team['member_contact_2'] ); ?>"></a>
										</li>
										<li>
											<a href="<?php echo esc_url( $team['link_3']['url'] ); ?>" class="<?php echo esc_attr( $team['member_contact_3'] ); ?>"></a>
										</li>
										<li>
											<a href="<?php echo esc_url( $team['link_4']['url'] ); ?>" class="<?php echo esc_attr( $team['member_contact_4'] ); ?>"></a>
										</li>
									</ul>
								</div>
							</div>
						</div>
					<?php } ?>
				</div>

				<div class="teammember-slider-dots"></div>
			</div>
		</div>
		<?php
	}
}

Plugin::instance()->widgets_manager->register_widget_type( new Azo_Team_Member() );
