<?php
/**
 * Elementor Widget
 *
 * @package azo Elementor Widget
 */

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Azo_Tab_Content
 */
class Azo_Tab_Content extends Widget_Base {
	/**
	 * Name
	 */
	public function get_name() {
		return 'azo-tab-content';
	}

	/**
	 * Title
	 */
	public function get_title() {
		return esc_html__( 'Azo Tab Of Content', 'azo' );
	}

	/**
	 * Icon
	 */
	public function get_icon() {
		return 'eicon-tabs';
	}

	/** SCRIPTS. */
	public function get_script_depends() {
		return array( 'azo-tab-content' );
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return array( 'azo-theme' );
	}

	/**
	 *  Get Saved Widgets
	 *
	 *  @param string $type page.
	 *  @since 0.0.1
	 *  @return string
	 */
	public function get_saved_data( $type = 'page' ) {
		$saved_widgets = $this->get_post_template( $type );
		$options[-1]   = esc_html__( 'Select', 'azo' );
		if ( count( $saved_widgets ) ) {
			foreach ( $saved_widgets as $saved_row ) {
				$options[ $saved_row['id'] ] = $saved_row['name'];
			}
		} else {
			$options['no_template'] = esc_html__( 'It seems that, you have not saved any template yet.', 'azo' );
		}
		return $options;
	}

	/**
	 *  Get Templates based on category
	 *
	 *  @param string $type Type.
	 *  @since 0.0.1
	 *  @return string
	 */
	public function get_post_template( $type = 'page' ) {
		$posts = get_posts(
			array(
				'post_type'      => 'elementor_library',
				'orderby'        => 'title',
				'order'          => 'ASC',
				'posts_per_page' => '-1',
				'tax_query'      => array( // phpcs:ignore
					array(
						'taxonomy' => 'elementor_library_type',
						'field'    => 'slug',
						'terms'    => $type,
					),
				),
			)
		);

		$templates = array();

		foreach ( $posts as $post ) {
			$templates[] = array(
				'id'   => $post->ID,
				'name' => $post->post_title,
			);
		}

		return $templates;
	}

	/**
	 * Registers all controls.
	 *
	 * @since 0.0.1
	 * @access protected
	 */
	protected function register_controls() { // phpcs:ignore
		// Rbs heading section starts.
		$this->start_controls_section(
			'rbs_section_content_1',
			array(
				'label' => esc_html__( 'Content', 'azo' ),
			)
		);

		$repeater = new \Elementor\Repeater();

		// Title tab option.
		$repeater->add_control(
			'list_title_tab',
			array(
				'label'   => esc_html__( 'Tab Option', 'azo' ),
				'type'    => Controls_Manager::TEXT,
				'default' => esc_html__( 'Our Mission', 'azo' ),
			)
		);

		// Rbs content section 1.
		$repeater->add_control(
			'rbs_select_section',
			array(
				'label'   => esc_html__( 'Section', 'azo' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'saved_rows',
				'options' => $this->get_content_type(),
			)
		);

		// Rbs content section 1 - saved rows.
		$repeater->add_control(
			'section_saved_rows',
			array(
				'label'   => esc_html__( 'Select Section', 'azo' ),
				'type'    => Controls_Manager::SELECT,
				'options' => $this->get_saved_data( 'section' ),
				'default' => '-1',
			)
		);

		$this->add_control(
			'list',
			array(
				'label'       => esc_html__( 'Repeater List', 'azo' ),
				'type'        => Controls_Manager::REPEATER,
				'fields'      => $repeater->get_controls(),
				'title_field' => '{{{ list_title_tab }}}',
				'default'     => array(
					array(
						array(),
						array(),
						array(),
					),
				),
			)
		);

		// Rbs heading section ends.
		$this->end_controls_section();

		$this->start_controls_section(
			'section_content_style',
			array(
				'label' => esc_html__( 'Style', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			array(
				'name'     => 'title_typography',
				'selector' => '{{WRAPPER}} .tab-container .list-content span',
			)
		);

		$this->add_control(
			'text_align',
			array(
				'label'     => esc_html__( 'Alignment', 'azo' ),
				'type'      => Controls_Manager::CHOOSE,
				'options'   => array(
					'flex-start' => array(
						'title' => esc_html__( 'Left', 'azo' ),
						'icon'  => 'fa fa-align-left',
					),
					'center'     => array(
						'title' => esc_html__( 'Center', 'azo' ),
						'icon'  => 'fa fa-align-center',
					),
					'flex-end'   => array(
						'title' => esc_html__( 'Right', 'azo' ),
						'icon'  => 'fa fa-align-right',
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .tab-container' => 'justify-content:{{VALUE}};',
				),
			)
		);

		$this->add_control(
			'title_color',
			array(
				'label'     => esc_html__( 'Color title', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .tab-container .list-content' => 'color: {{VALUE}}',
				),
				'default'   => '#404080',
				'separator' => 'before',
			)
		);

		$this->add_control(
			'title_active_color',
			array(
				'label'     => esc_html__( 'Color title active', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .tab-container .list-content.current' => 'color: {{VALUE}}',
					'{{WRAPPER}} .tab-container .list-content:hover'   => 'color: {{VALUE}}',
					'{{WRAPPER}} .tab-container .list-content:before'  => 'background-color: {{VALUE}}',
				),
				'default'   => '#ffb700',
				'separator' => 'before',
			)
		);

		$this->add_responsive_control(
			'tab_spacer',
			array(
				'label'      => esc_html__( 'Space', 'azo' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'min' => 0,
					'max' => 1000,
				),
				'default'    => array(
					'unit' => 'px',
					'size' => 50,
				),
				'selectors'  => array(
					'{{WRAPPER}} .tab-container' => 'margin-bottom:{{SIZE}}{{UNIT}};',
				),
				'separator'  => 'before',
			)
		);
		$this->end_controls_section();
	}

	/**
	 * Render content type list.
	 */
	public function get_content_type() {
		$content_type = array(
			'saved_rows' => esc_html__( 'Saved Section', 'azo' ),
		);

		return $content_type;
	}

	/**
	 * Render button widget classes names.
	 *
	 * @param string $item page.
	 * @param string $node_id page.
	 * @param string $section page.
	 */
	public function get_modal_content( $item, $node_id, $section ) {
		$normal_content = $this->get_settings_for_display( 'section_content' );
		$content_type   = $item[ $section ];
		if ( 'rbs_select_section' === $section ) {
			switch ( $content_type ) {
				case 'saved_rows':
					return \Elementor\Plugin::$instance->frontend->get_builder_content_for_display( $item['section_saved_rows'] );
				default:
					break;
			}
		}
	}

	/**
	 * Render
	 */
	protected function render() {
		$settings  = $this->get_settings();
		$node_id   = $this->get_id();
		$is_editor = \Elementor\Plugin::instance()->editor->is_edit_mode();
		?>
		<div class="tabs">
			<ul class="nav-tabs tab-container" role="tablist">
				<?php
				if ( $settings['list'] ) {
					$i         = wp_rand( 0, 100 );
					$i_content = $i;

					foreach ( $settings['list'] as $key => $item ) {
						$active = '';
						if ( '0' === $key ) {
							$active = 'active';
						}
						?>
						<li data-select="tab-<?php echo esc_attr( $item['_id'] ); ?>" class="list-content ui-tabs-<?php echo esc_attr( $active ); ?>">
							<span class="nav-item nav-link">
								<?php echo esc_html( $item['list_title_tab'] ); ?> 
							</span>
						</li>
						<?php
						$i++;
					}
				}
				?>
			</ul>

			<div class="tab-content-area">
				<?php
				foreach ( $settings['list'] as $key => $item ) {
					?>
					<div class="tab-content-item" id="tab-<?php echo esc_attr( $item['_id'] ); ?>">
						<p>
							<?php echo do_shortcode( $this->get_modal_content( $item, $item['_id'], 'rbs_select_section' ) ); ?>
						</p>
					</div>
				<?php } ?>
			</div>
		</div>
		<!--tababble-->
		<?php
	}
}

Plugin::instance()->widgets_manager->register_widget_type( new Azo_Tab_Content() );
