<?php
/**
 * Products tabs map.
 *
 * @package azo
 */

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

if ( ! defined( 'WC_PLUGIN_FILE' ) ) {
	return;
}

/**
 * Elementor widget that inserts an embeddable content into the page, from any given URL.
 *
 * @since 1.0.0
 */
class Azo_Products_Tabs extends Widget_Base {
	/**
	 * Get widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'azo-products-tabs';
	}
	/**
	 * Get widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Products tabs', 'azo' );
	}

	/**
	 * Get widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-tabs';
	}

	/**
	 * Get categories
	 */
	public function get_categories() {
		return array( 'azo-theme' );
	}

	/**
	 * Controls
	 */
	protected function register_controls() { // phpcs:ignore
		$this->general();
		$this->filter();
		$this->product();
	}

	/**
	 * General
	 */
	private function general() {
		/**
		 * General settings.
		 */
		$this->start_controls_section(
			'general_content_section',
			array(
				'label' => esc_html__( 'General', 'azo' ),
			)
		);

		$this->add_control(
			'title',
			array(
				'label'   => esc_html__( 'Tabs title', 'azo' ),
				'type'    => Controls_Manager::TEXT,
				'default' => 'Title text example',
			)
		);

		$this->add_control(
			'layout',
			array(
				'label'   => __( 'Layout', 'azo' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'grid',
				'options' => array(
					'grid'     => __( 'Grid', 'azo' ),
					'carousel' => __( 'Carousel', 'azo' ),
				),
			)
		);

		$this->add_control(
			'arrow',
			array(
				'label'        => esc_html__( 'Show Arrows', 'azo' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'no',
				'label_on'     => esc_html__( 'No', 'azo' ),
				'label_off'    => esc_html__( 'Yes', 'azo' ),
				'return_value' => 'yes',
				'condition'    => array(
					'layout' => 'carousel',
				),
			)
		);

		$this->add_control(
			'active_underline',
			array(
				'label'        => esc_html__( 'Button Active Underline', 'azo' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', 'azo' ),
				'label_off'    => esc_html__( 'Hide', 'azo' ),
				'return_value' => 'yes',
				'default'      => 'no',
			)
		);

		$this->add_control(
			'button_viewall',
			array(
				'label'        => esc_html__( 'Button View All', 'azo' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', 'azo' ),
				'label_off'    => esc_html__( 'Hide', 'azo' ),
				'return_value' => 'yes',
				'default'      => 'no',
			)
		);

		// Sub Title Heading.
		$this->add_control(
			'widget_sub_title',
			array(
				'label'     => esc_html__( 'Sub title', 'azo' ),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'default'   => esc_html__( 'All Products', 'azo' ),
				'condition' => array(
					'button_viewall' => 'yes',
				),
			)
		);

		$this->add_control(
			'link',
			array(
				'label'     => __( 'Link', 'azo' ),
				'type'      => Controls_Manager::URL,
				'dynamic'   => array(
					'active' => true,
				),
				'default'   => array(
					'url' => '',
				),
				'condition' => array(
					'button_viewall' => 'yes',
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'general',
			array(
				'label' => esc_html__( 'Tabs', 'azo' ),
			)
		);

		$repeater = new Repeater();

		$repeater->add_control(
			'title',
			array(
				'label'   => __( 'Title', 'azo' ),
				'type'    => Controls_Manager::TEXT,
				'default' => __( 'List Title', 'azo' ),
			)
		);

		$repeater->add_control(
			'data',
			array(
				'label'     => __( 'Source', 'azo' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => 'select',
				'separator' => 'before',
				'options'   => array(
					'select'    => __( 'Manual Selection', 'azo' ),
					'latest'    => __( 'Latest Products', 'azo' ),
					'featured'  => __( 'Featured', 'azo' ),
					'best-sell' => __( 'Best Sellers', 'azo' ),
					'sale'      => __( 'On Sale', 'azo' ),
				),
			)
		);

		$repeater->add_control(
			'cat_ids',
			array(
				'label'     => esc_html__( 'Categories', 'azo' ),
				'type'      => 'autocomplete',
				'query'     => array(
					'type' => 'term',
					'name' => 'product_cat',
				),
				'condition' => array(
					'data' => 'select',
				),
			)
		);

		$repeater->add_control(
			'product_ids',
			array(
				'label'     => esc_html__( 'Products', 'azo' ),
				'type'      => 'autocomplete',
				'query'     => array(
					'type' => 'post_type',
					'name' => 'product',
				),
				'condition' => array(
					'data' => 'select',
				),
			)
		);

		$repeater->add_control(
			'ex_cat_ids',
			array(
				'label'     => esc_html__( 'Exclude Categories', 'azo' ),
				'type'      => 'autocomplete',
				'query'     => array(
					'type' => 'term',
					'name' => 'product_cat',
				),
				'condition' => array(
					'data!' => 'latest',
				),
			)
		);

		$repeater->add_control(
			'ex_product_ids',
			array(
				'label'     => esc_html__( 'Exclude Products', 'azo' ),
				'type'      => 'autocomplete',
				'query'     => array(
					'type' => 'post_type',
					'name' => 'product',
				),
				'condition' => array(
					'data!' => 'latest',
				),
			)
		);

		$repeater->add_control(
			'total',
			array(
				'label'     => esc_html__( 'Total Products', 'azo' ),
				'type'      => Controls_Manager::NUMBER,
				'default'   => 4,
				'min'       => 1,
				'max'       => 100,
				'step'      => 1,
				'separator' => 'before',
			)
		);

		$repeater->add_control(
			'col',
			array(
				'type'    => Controls_Manager::SELECT,
				'label'   => esc_html__( 'Columns', 'azo' ),
				'default' => 4,
				'options' => array(
					1 => 1,
					2 => 2,
					3 => 3,
					4 => 4,
					5 => 5,
					6 => 6,
				),
			)
		);

		$repeater->add_control(
			'orderby',
			array(
				'label'     => esc_html__( 'Order By', 'azo' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => 'name',
				'condition' => array(
					'data!' => array(
						'best-sell',
						'latest',
					),
				),
				'options'   => array(
					'ID'     => esc_html__( 'ID', 'azo' ),
					'name'   => esc_html__( 'Name', 'azo' ),
					'title'  => esc_html__( 'Title', 'azo' ),
					'price'  => esc_html__( 'Price', 'azo' ),
					'rating' => esc_html__( 'Rating', 'azo' ),
					'date'   => esc_html__( 'Date', 'azo' ),
					'rand'   => esc_html__( 'Random', 'azo' ),
				),
			)
		);

		$repeater->add_control(
			'order',
			array(
				'label'     => esc_html__( 'Order', 'azo' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => 'ASC',
				'condition' => array(
					'data!' => 'latest',
				),
				'options'   => array(
					'ASC'  => esc_html__( 'ASC', 'azo' ),
					'DESC' => esc_html__( 'DESC', 'azo' ),
				),
			)
		);

		$this->add_control(
			'list',
			array(
				'label'       => __( 'Product Tab', 'azo' ),
				'type'        => Controls_Manager::REPEATER,
				'fields'      => $repeater->get_controls(),
				'default'     => array(
					array(
						'title' => __( 'Featured', 'azo' ),
						'data'  => 'featured',
					),
					array(
						'title' => __( 'Best Sellers', 'azo' ),
						'data'  => 'best-sell',
					),
				),
				'title_field' => '{{{ title }}}',
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Product
	 */
	private function product() {

		$this->start_controls_section(
			'product',
			array(
				'label' => __( 'Product', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'product_title',
			array(
				'label' => __( 'Title', 'azo' ),
				'type'  => Controls_Manager::HEADING,
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'product_style_title_typo',
				'label'    => esc_html__( 'Typography', 'azo' ),
				'selector' => '{{WRAPPER}} .woocommerce-loop-product__title',
			)
		);

		// Title Spacing.
		$this->add_responsive_control(
			'title_spacing',
			array(
				'label'           => __( 'Spacing', 'azo' ),
				'type'            => Controls_Manager::SLIDER,
				'range'           => array(
					'px' => array(
						'max' => 200,
					),
				),
				'devices'         => array(
					'desktop',
					'tablet',
					'mobile',
				),
				'desktop_default' => array(
					'size' => 0,
					'unit' => 'px',
				),
				'tablet_default'  => array(
					'size' => 0,
					'unit' => 'px',
				),
				'mobile_default'  => array(
					'size' => 0,
					'unit' => 'px',
				),
				'selectors'       => array(
					'{{WRAPPER}} .azo-products-tab-content .woocommerce-loop-product__title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				),
			)
		);

		// TAB START.
		$this->start_controls_tabs( 'product_style_tabs' );

		// Normal.
		$this->start_controls_tab(
			'product_style_normal',
			array(
				'label' => __( 'Normal', 'azo' ),
			)
		);

		// Color.
		$this->add_control(
			'product_style_title_color',
			array(
				'label'     => __( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .woocommerce-loop-product__title' => 'color: {{VALUE}};',
				),
			)
		);

		// END NORMAL.
		$this->end_controls_tab();

		// HOVER.
		$this->start_controls_tab(
			'product_style_hover',
			array(
				'label' => __( 'Hover', 'azo' ),
			)
		);

		// Hover color.
		$this->add_control(
			'product_style_color',
			array(
				'label'     => __( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} a:hover .woocommerce-loop-product__title' => 'color: {{VALUE}};',
				),
			)
		);

		// TAB END.
		$this->end_controls_tab();
		$this->end_controls_tabs();

		// Price.
		$this->add_control(
			'product_price',
			array(
				'label'     => __( 'Price', 'azo' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		// Color Price.
		$this->add_control(
			'product_price_color',
			array(
				'label'     => __( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-products-tab-content .price ins span, {{WRAPPER}} .azo-products-tab-content .price span' => 'color: {{VALUE}};',
				),
			)
		);

		// Price Typography.
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'product_price_typo',
				'label'    => esc_html__( 'Typography', 'azo' ),
				'selector' => '{{WRAPPER}} .azo-products-tab-content .price ins span, {{WRAPPER}} .azo-products-tab-content .price span',
			)
		);

		// Regular Price.
		$this->add_control(
			'product_regular_price',
			array(
				'label'     => __( 'Sale Price', 'azo' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		// Color Regular Price.
		$this->add_control(
			'product_regular_price_color',
			array(
				'label'     => __( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-products-tab-content .price del span, {{WRAPPER}} .azo-products-tab-content .price del' => 'color: {{VALUE}};',
				),
			)
		);

		// Regular Price Typography.
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'product_regular_price_typo',
				'label'    => esc_html__( 'Typography', 'azo' ),
				'selector' => '{{WRAPPER}} .azo-products-tab-content .price del span',
			)
		);

		// Style Button.
		$this->add_control(
			'product_button',
			array(
				'label'     => __( 'Button', 'azo' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		// Button Typography.
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'product_button_typo',
				'label'    => esc_html__( 'Typography', 'azo' ),
				'selector' => '{{WRAPPER}} .azo-add-to-cart-btn',
			)
		);

		// TAB START.
		$this->start_controls_tabs( 'atcbutton_style_tabs' );

		// Normal.
		$this->start_controls_tab(
			'atcbutton_style_normal',
			array(
				'label' => __( 'Normal', 'azo' ),
			)
		);

		// Color Button.
		$this->add_control(
			'atcbutton_button_color',
			array(
				'label'     => __( 'Color Button', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-add-to-cart-btn' => 'color: {{VALUE}};',
				),
			)
		);

		// Background Color Button.
		$this->add_control(
			'atcbutton_button_bg_color',
			array(
				'label'     => __( 'Background Color Button', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-add-to-cart-btn' => 'background-color: {{VALUE}};',
				),
			)
		);

		// END NORMAL.
		$this->end_controls_tab();

		// HOVER.
		$this->start_controls_tab(
			'atcbutton_style_hover',
			array(
				'label' => __( 'Hover', 'azo' ),
			)
		);

		// Color Hover Button.
		$this->add_control(
			'atcbutton_button_hover_color',
			array(
				'label'     => __( 'Color Hover Button', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-add-to-cart-btn:hover' => 'color: {{VALUE}};',
				),
			)
		);

		// Background Color Button.
		$this->add_control(
			'atcbutton_button_bg_hover_color',
			array(
				'label'     => __( 'Background Hover Color Button', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-add-to-cart-btn:hover' => 'background-color: {{VALUE}};',
				),
			)
		);

		// TAB END.
		$this->end_controls_tab();
		$this->end_controls_tabs();

		// Style Rate.
		$this->add_control(
			'product_rate',
			array(
				'label'     => __( 'Rate', 'azo' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		// Color Rate.
		$this->add_control(
			'product_rate_color',
			array(
				'label'     => __( 'Color Rate', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .star-rating > span:before' => 'color: {{VALUE}};',
				),
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Tabs
	 */
	private function filter() {

		$this->start_controls_section(
			'filter',
			array(
				'label' => __( 'Filter', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'filter_heading',
			array(
				'label' => __( 'Tab Heading', 'azo' ),
				'type'  => Controls_Manager::HEADING,
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'style_title_heading_typo',
				'label'    => esc_html__( 'Typography Heading', 'azo' ),
				'selector' => '{{WRAPPER}} .tabs-name.title .tabs-text',
			)
		);

		// Color.
		$this->add_control(
			'heading_title_color',
			array(
				'label'     => __( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .tabs-name.title .tabs-text' => 'color: {{VALUE}};',
				),
			)
		);

		// Button View All.
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'      => 'style_buttonviewall_typo',
				'label'     => esc_html__( 'Typography Button', 'azo' ),
				'selector'  => '{{WRAPPER}} .tabs-button-view a',
				'condition' => array(
					'button_viewall' => 'yes',
				),
			)
		);

		// Color.
		$this->add_control(
			'buttonviewall_color',
			array(
				'label'     => __( 'Color Button', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .tabs-button-view a' => 'color: {{VALUE}};',
				),
				'condition' => array(
					'button_viewall' => 'yes',
				),
			)
		);

		$this->add_control(
			'buttonviewall_hover_color',
			array(
				'label'     => __( 'Color Button Hover', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .tabs-button-view a:hover' => 'color: {{VALUE}};',
				),
				'condition' => array(
					'button_viewall' => 'yes',
				),
			)
		);

		$this->add_control(
			'selected_icon',
			array(
				'label'            => __( 'Icon', 'azo' ),
				'type'             => Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'default'          => array(
					'value'   => 'fas fa-star',
					'library' => 'fa-solid',
				),
				'condition'        => array(
					'button_viewall' => 'yes',
				),
			)
		);

		$this->add_responsive_control(
			'size',
			array(
				'label'     => __( 'Size', 'azo' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min' => 6,
						'max' => 300,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .azo-icon i'   => 'font-size: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .azo-icon svg' => 'width: {{SIZE}}{{UNIT}}; height: auto',
				),
				'condition' => array(
					'button_viewall' => 'yes',
				),
			)
		);

		// Typo icon.
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'      => 'style_icon_typo',
				'label'     => esc_html__( 'Typography Icon', 'azo' ),
				'selector'  => '{{WRAPPER}} .azo-icon',
				'condition' => array(
					'button_viewall' => 'yes',
				),
			)
		);

		$this->add_responsive_control(
			'rotate',
			array(
				'label'          => __( 'Rotate', 'azo' ),
				'type'           => Controls_Manager::SLIDER,
				'size_units'     => array( 'deg' ),
				'default'        => array(
					'size' => 0,
					'unit' => 'deg',
				),
				'tablet_default' => array(
					'unit' => 'deg',
				),
				'mobile_default' => array(
					'unit' => 'deg',
				),
				'selectors'      => array(
					'{{WRAPPER}} .azo-icon i, {{WRAPPER}} .azo-icon svg' => 'transform: rotate({{SIZE}}{{UNIT}});',
				),
				'condition'      => array(
					'button_viewall' => 'yes',
				),
			)
		);

		// Icon Spacing.
		$this->add_responsive_control(
			'button_icon_spacing',
			array(
				'label'           => __( 'Icon Button Space Left', 'azo' ),
				'type'            => Controls_Manager::SLIDER,
				'range'           => array(
					'px' => array(
						'max' => 50,
					),
				),
				'devices'         => array(
					'desktop',
					'tablet',
					'mobile',
				),
				'desktop_default' => array(
					'size' => 29,
					'unit' => 'px',
				),
				'tablet_default'  => array(
					'size' => 0,
					'unit' => 'px',
				),
				'mobile_default'  => array(
					'size' => 0,
					'unit' => 'px',
				),
				'selectors'       => array(
					'{{WRAPPER}} .azo-icon' => 'margin-left: {{SIZE}}{{UNIT}};',
				),
				'condition'       => array(
					'button_viewall' => 'yes',
				),
			)
		);

		// Border Color.
		$this->add_control(
			'heading_title_border_color',
			array(
				'label'     => __( 'Border Bottom Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-products-tab-widget .azo-products-tab-head' => 'border-bottom-color: {{VALUE}};',
				),
			)
		);

		// Border Color Heading.
		$this->add_control(
			'heading_title_heading_border_color',
			array(
				'label'     => __( 'Heading Border Bottom Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-products-tab-widget .azo-products-tab-head .azo-products-tab-head-buttons .tabs-name.title:after' => 'background-color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'filter_tab',
			array(
				'label'     => __( 'Tab', 'azo' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		// Alignment.
		$this->add_control(
			'alignment',
			array(
				'type'      => Controls_Manager::CHOOSE,
				'label'     => esc_html__( 'Alignment', 'azo' ),
				'options'   => array(
					'flex-start'    => array(
						'title' => esc_html__( 'Left', 'azo' ),
						'icon'  => 'fa fa-align-left',
					),
					'center'        => array(
						'title' => esc_html__( 'Center', 'azo' ),
						'icon'  => 'fa fa-align-center',
					),
					'flex-end'      => array(
						'title' => esc_html__( 'Right', 'azo' ),
						'icon'  => 'fa fa-align-right',
					),
					'space-between' => array(
						'title' => esc_html__( 'Space Between', 'azo' ),
						'icon'  => 'fa fa-align-justify',
					),
				),
				'default'   => '',
				'selectors' => array(
					'{{WRAPPER}} .azo-products-tab-head-buttons' => 'justify-content: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'heading_margin',
			array(
				'label'              => __( 'Margin', 'azo' ),
				'type'               => Controls_Manager::DIMENSIONS,
				'size_units'         => array( 'px', 'em' ),
				'allowed_dimensions' => array( 'top', 'bottom' ),
				'selectors'          => array(
					'{{WRAPPER}} .azo-products-tab-head' => 'margin: {{TOP}}{{UNIT}} 0px {{BOTTOM}}{{UNIT}} 0px;',
				),
			)
		);

		$this->add_control(
			'heading_item_padding',
			array(
				'label'      => __( 'Padding', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .azo-products-tab-btn' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			array(
				'name'     => 'heading_border',
				'label'    => __( 'Border', 'azo' ),
				'selector' => '{{WRAPPER}} .azo-products-tab-head',
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'heading_typo',
				'label'    => __( 'Typography', 'azo' ),
				'selector' => '{{WRAPPER}} .azo-products-tab-btn',
			)
		);

		$this->add_control(
			'filter_list',
			array(
				'label'     => __( 'Tab List', 'azo' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			array(
				'name'     => 'list_border',
				'label'    => __( 'Border', 'azo' ),
				'selector' => '{{WRAPPER}} .azo-products-tab-btn',
			)
		);

		// TAB START.
		$this->start_controls_tabs( 'style_tabs' );

		// Normal.
		$this->start_controls_tab(
			'style_tabs_normal',
			array(
				'label' => __( 'Normal', 'azo' ),
			)
		);

		// Color.
		$this->add_control(
			'tabs_title_color',
			array(
				'label'     => __( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-products-tab-btn' => 'color: {{VALUE}};',
				),
			)
		);

		// Background.
		$this->add_control(
			'tabs_background_color',
			array(
				'label'     => __( 'Background', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-products-tab-btn' => 'background: {{VALUE}};',
				),
			)
		);

		// Border.
		$this->add_control(
			'style_title_border',
			array(
				'label'     => __( 'Border', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-products-tab-btn' => 'border-color: {{VALUE}};',
				),
			)
		);

		// END NORMAL.
		$this->end_controls_tab();

		// HOVER.
		$this->start_controls_tab(
			'style_tabs_hover',
			array(
				'label' => __( 'Hover', 'azo' ),
			)
		);

		// Hover color.
		$this->add_control(
			'style_title_hover_color',
			array(
				'label'     => __( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-products-tab-btn:hover' => 'color: {{VALUE}};',
				),
			)
		);

		// Hover Background.
		$this->add_control(
			'tabs_background_hover_color',
			array(
				'label'     => __( 'Background', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-products-tab-btn:hover' => 'background: {{VALUE}};',
				),
			)
		);

		// Hover border.
		$this->add_control(
			'style_title_hover_border',
			array(
				'label'     => __( 'Border', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-products-tab-btn:hover' => 'border-color: {{VALUE}};',
				),
			)
		);

		// TAB END.
		$this->end_controls_tab();

		// HOVER.
		$this->start_controls_tab(
			'style_tabs_active',
			array(
				'label' => __( 'Active', 'azo' ),
			)
		);

		// Active color.
		$this->add_control(
			'style_title_active_color',
			array(
				'label'     => __( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-products-tab-btn.active' => 'color: {{VALUE}};',
				),
			)
		);

		// Active Background.
		$this->add_control(
			'tabs_background_active_color',
			array(
				'label'     => __( 'Background', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-products-tab-btn.active' => 'background: {{VALUE}};',
				),
			)
		);

		// Active border.
		$this->add_control(
			'style_title_active_border',
			array(
				'label'     => __( 'Border', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-products-tab-btn.active' => 'border-color: {{VALUE}};',
				),
			)
		);

		// TAB END.
		$this->end_controls_tab();
		$this->end_controls_tabs();

		$this->end_controls_section();
	}

	/**
	 * Product tab render
	 */
	public function product_tab_render() {
		$settings = $this->get_settings_for_display();
		$list     = $settings['list'];
		$response = array();

		foreach ( $list as $k ) {
			$args = array(
				'post_type'      => 'product',
				'post_status'    => 'publish',
				'posts_per_page' => $k['total'],
				'order'          => $k['order'],
			);

			// Category ids.
			$cat_ids    = $k['cat_ids'];
			$ex_cat_ids = $k['ex_cat_ids'];

			// Product ids.
			$product_ids    = $k['product_ids'];
			$ex_product_ids = $k['ex_product_ids'];

			switch ( $k['data'] ) {
				case 'select':
				default:
					if ( ! empty( $cat_ids ) ) {
						$args['tax_query'] = array( // phpcs:ignore
							array(
								'taxonomy' => 'product_cat',
								'field'    => 'term_id',
								'operator' => 'IN',
								'terms'    => $cat_ids,
							),
						);
					}

					if ( ! empty( $product_ids ) ) {
						$args['post__in'] = $product_ids;
					}

					if ( ! empty( $ex_product_ids ) ) {
						$args['post__not_in'] = $ex_product_ids;
					}

					break;
				case 'latest':
					$args = array(
						'post_type'   => 'product',
						'post_status' => 'publish',
						'order'       => 'DESC',
						'orderby'     => 'ID',
					);

					break;
				case 'featured':
					$args['tax_query'] = array( // phpcs:ignore
						'relation' => 'AND',
						array(
							'taxonomy' => 'product_visibility',
							'field'    => 'name',
							'terms'    => 'featured',
							'operator' => 'IN',
						),
					);

					if ( ! empty( $ex_cat_ids ) ) {
						$args['tax_query'] = array( // phpcs:ignore
							'taxonomy' => 'product_cat',
							'field'    => 'term_id',
							'terms'    => $ex_cat_ids,
							'operator' => 'NOT IN',
						);
					}

					if ( ! empty( $ex_product_ids ) ) {
						$args['post__not_in'] = $ex_product_ids;
					}

					break;
				case 'best-sell':
					$args['meta_key'] = 'total_sales'; // phpcs:ignore

					if ( ! empty( $ex_cat_ids ) ) {
						$args['tax_query'][] = array(
							'taxonomy' => 'product_cat',
							'field'    => 'term_id',
							'terms'    => $ex_cat_ids,
							'operator' => 'NOT IN',
						);
					}

					if ( ! empty( $ex_product_ids ) ) {
						$args['post__not_in'] = $ex_product_ids;
					}

					break;
				case 'sale':
					$args['meta_query'] = array( // phpcs:ignore
						'relation' => 'OR',
						array(
							'key'     => '_sale_price',
							'value'   => 0,
							'compare' => '>',
							'type'    => 'numeric',
						),
						array(
							'key'     => '_min_variation_sale_price',
							'value'   => 0,
							'compare' => '>',
							'type'    => 'numeric',
						),
					);

					$args['tax_query'] = array( // phpcs:ignore
						array(
							'taxonomy' => 'product_cat',
							'field'    => 'term_id',
							'terms'    => $ex_cat_ids,
							'operator' => 'NOT IN',
						),
					);

					if ( ! empty( $ex_product_ids ) ) {
						$args['post__not_in'] = $ex_product_ids;
					}

					break;
			}

			switch ( $k['orderby'] ) {
				case 'price':
					$args['orderby']  = 'meta_value_num';
					$args['meta_key'] = '_price'; // phpcs:ignore
					break;
				case 'rating':
					$args['orderby']  = 'meta_value_num';
					$args['meta_key'] = '_wc_average_rating'; // phpcs:ignore
					break;
				default:
					if ( 'best-sell' === $k['data'] ) {
						$args['orderby'] = 'meta_value_num';
					} else {
						$args['orderby'] = $k['orderby'];
					}
					break;
			}

			array_push( $response, $args );
		}

		return $response;
	}

	/**
	 * Render
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
		$list     = $settings['list'];
		if ( empty( $list ) ) {
			return;
		}

		$data        = $this->product_tab_render();
		$arrow_left  = apply_filters( 'azo_product_tab_carousel_arrow_left_icon', 'ti-angle-left' );
		$arrow_right = apply_filters( 'azo_product_tab_carousel_arrow_right_icon', 'ti-angle-right' );

		$list_first = array_shift( $list );
		$data_first = array_shift( $data );

		// Detect first slider init.
		$has_slider = ! empty( $data_first['posts_per_page'] ) && intval( $data_first['posts_per_page'] ) > intval( $list_first['col'] ) ? 'has-slider' : '';

		if ( 'yes' === $settings['active_underline'] ) {
			$undeline = 'underline';
		} else {
			$undeline = '';
		}

		$href = '#';
		$attr = '';

		if ( ! empty( $settings['link']['url'] ) ) {
			$href = $settings['link']['url'];

			if ( 'on' === $settings['link']['is_external'] ) {
				$attr .= ' target="_blank"';
			}

			if ( 'on' === $settings['link']['nofollow'] ) {
				$attr .= ' rel="nofollow"';
			}
		}

		$migrated = isset( $settings['__fa4_migrated']['selected_icon'] );
		$is_new   = empty( $settings['icon'] ) && Icons_Manager::is_migration_allowed();
		?>

		<div class="azo-products-tab-widget" data-layout="<?php echo esc_attr( $settings['layout'] ); ?>-layout">
			<div class="azo-products-tab-head">
				<div class="azo-products-tab-head-buttons <?php echo esc_attr( $undeline ); ?> layout-<?php echo esc_attr( $settings['layout'] ); ?>">
					<div class="tabs-heading">
						<?php if ( $settings['title'] ) { ?>
							<div class="tabs-name title">
								<span class="tabs-text">
									<?php echo wp_kses( $settings['title'], azo_get_allowed_html() ); ?>
								</span>
							</div>
						<?php } ?>

						<?php if ( 'yes' === $settings['button_viewall'] ) { ?>
							<div class="tabs-button-view">
								<?php if ( $settings['widget_sub_title'] ) { ?>
									<a href="<?php echo esc_url( $href ); ?>" <?php echo wp_kses( $attr, azo_get_allowed_html() ); ?>>
										<?php echo esc_html( $settings['widget_sub_title'] ); ?>

										<span class="azo-icon">
											<?php
											if ( $is_new || $migrated ) {
												Icons_Manager::render_icon( $settings['selected_icon'], array( 'aria-hidden' => 'true' ) );
											} else {
												?>
												<i <?php echo wp_kses( $this->get_render_attribute_string( 'icon' ), azo_get_allowed_html() ); ?>></i>
											<?php } ?>
										</span>
									</a>
								<?php } ?>
							</div>
						<?php } ?>
					</div>

					<div class="tabs-filter">
						<span class="azo-products-tab-btn ready active" data-id="<?php echo esc_attr( $list_first['_id'] ); ?>"><?php echo esc_html( $list_first['title'] ); ?></span>

						<?php
						if ( ! empty( $list ) ) {
							foreach ( $list as $v ) {
								?>
								<span class="azo-products-tab-btn" data-id="<?php echo esc_attr( $v['_id'] ); ?>"><?php echo esc_html( $v['title'] ); ?></span>
								<?php
							}
						}
						?>
					</div>
				</div>

				<?php
				if ( 'carousel' === $settings['layout'] ) {
					if ( 'yes' === $settings['arrow'] ) {
						?>
						<div class="azo-product-tab-carousel-arrows">
							<div class="azo-product-tab-arrows-container active <?php echo esc_attr( $has_slider ? '' : 'hidden' ); ?>" data-id="<?php echo esc_attr( $list_first['_id'] ); ?>">
								<span class="<?php echo esc_attr( $arrow_left ); ?>"></span>
								<span class="<?php echo esc_attr( $arrow_right ); ?>"></span>
							</div>

							<?php
							if ( ! empty( $list ) ) {
								foreach ( $list as $v ) {
									?>
									<div class="azo-product-tab-arrows-container" data-id="<?php echo esc_attr( $v['_id'] ); ?>">
										<span class="<?php echo esc_attr( $arrow_left ); ?>"></span>
										<span class="<?php echo esc_attr( $arrow_right ); ?>"></span>
									</div>
									<?php
								}
							}
							?>
						</div>
						<?php
					}
				}
				?>
			</div>

			<div class="azo-products-tab-body">
				<div class="azo-products-tab-content active <?php echo esc_attr( $has_slider ); ?>" data-columns="<?php echo esc_attr( $list_first['col'] ); ?>" data-id="<?php echo esc_attr( $list_first['_id'] ); ?>">
					<?php
					$query = new \WP_Query( $data_first );

					$product_columns = array(
						'products columns-6',
						'products columns-5',
						'products columns-4',
						'products columns-3',
						'products columns-2',
						'products columns-1',
					);

					ob_start();
					if ( $query->have_posts() ) {

						global $woocommerce_loop;

						$woocommerce_loop['columns'] = (int) $list_first['col'];

						woocommerce_product_loop_start();

						while ( $query->have_posts() ) {
							$query->the_post();

							wc_get_template_part( 'content', 'product' );
						}

						woocommerce_product_loop_end();

						woocommerce_reset_loop();
					}
					$content_loop = ob_get_clean();

					for ( $i = 0; $i < 6; $i++ ) {
						$str = strpos( $content_loop, $product_columns[ $i ] );

						if ( ! $str ) {
							continue;
						}

						if ( false !== $str ) {
							$content_loop = str_replace( $product_columns[ $i ], 'products columns-' . $list_first['col'], $content_loop );
							break;
						}
					}

					echo wp_kses( $content_loop, azo_get_allowed_html() );
					?>
				</div>

				<?php
				if ( ! empty( $list ) ) {
					foreach ( $list as $i => $j ) {
						?>
						<div class="azo-products-tab-content" data-columns="<?php echo esc_attr( $j['col'] ); ?>" data-id="<?php echo esc_attr( $j['_id'] ); ?>" data-query='<?php echo wp_json_encode( $data[ $i ] ); ?>'></div>
						<?php
					}
				}
				?>
			</div>
		</div>
		<?php
	}
}

Plugin::instance()->widgets_manager->register_widget_type( new Azo_Products_Tabs() );
