<?php
/**
 * Elementor Product Slider Widget
 *
 * @package azo
 */

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

if ( ! defined( 'WC_PLUGIN_FILE' ) ) {
	return;
}

/**
 * Class for azo elementor product slider widget.
 */
class Azo_Products_Slider extends \Elementor\Widget_Base {
	/**
	 * Category
	 */
	public function get_categories() {
		return array( 'azo-theme' );
	}

	/**
	 * Name
	 */
	public function get_name() {
		return 'azo-product-slider';
	}

	/**
	 * Gets the title.
	 */
	public function get_title() {
		return __( 'Product Slider', 'azo' );
	}

	/**
	 * Gets the icon.
	 */
	public function get_icon() {
		return 'eicon-posts-carousel';
	}

	/**
	 * Retrieve the list of scripts the widget depended on
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return array( 'azo-product-slider' );
	}

	/**
	 * Controls
	 */
	protected function register_controls() { //phpcs:ignore
		$this->general();
		$this->section_query();
		$this->product();
	}

	/**
	 * General
	 */
	private function general() {
		$this->start_controls_section(
			'section_filter_product',
			array(
				'label' => esc_html__( 'Product Slider', 'azo' ),
			)
		);

		$this->add_control(
			'title',
			array(
				'label'   => __( 'Title', 'azo' ),
				'type'    => Controls_Manager::TEXT,
				'default' => 'Lastest Products',
			)
		);

		$this->add_control(
			'class_slider',
			array(
				'label'   => esc_html__( 'Choose Slider', 'azo' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'1' => esc_html__( 'Slider 1', 'azo' ),
					'2' => esc_html__( 'Slider 2', 'azo' ),
					'3' => esc_html__( 'Slider 3', 'azo' ),
				),
				'default' => '1',
			)
		);

		// Slides to show.
		$this->add_responsive_control(
			'slides_to_show',
			array(
				'label'                 => esc_html__( 'Slides to show', 'azo' ),
				'type'                  => Controls_Manager::NUMBER,
				'min'                   => 1,
				'max'                   => 5,
				'step'                  => 1,
				'devices'               => array( 'desktop', 'tablet', 'mobile' ),
				'slides_to_show_mobile' => 1,
				'slides_to_show_tablet' => 2,
				'slides_to_show'        => 4,
			)
		);

		$this->add_control(
			'style_arrow',
			array(
				'label'   => esc_html__( 'Arrow Preset', 'azo' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'arrowpreset1' => esc_html__( 'Arrow Preset 1', 'azo' ),
					'arrowpreset2' => esc_html__( 'Arrow Preset 2', 'azo' ),
				),
				'default' => 'arrowpreset1',
			)
		);

		$this->add_control(
			'rows',
			array(
				'label'   => esc_html__( 'Rows', 'azo' ),
				'type'    => Controls_Manager::NUMBER,
				'default' => 1,
			)
		);

		$this->add_control(
			'stock_bar',
			array(
				'label'        => esc_html__( 'Show Stock Bar', 'azo' ),
				'type'         => Controls_Manager::SWITCHER,
				'separator'    => 'before',
				'default'      => '',
				'on'           => esc_html__( 'No', 'azo' ),
				'off'          => esc_html__( 'Yes', 'azo' ),
				'return_value' => 'yes',
			)
		);

		$this->add_control(
			'arrow',
			array(
				'label'        => esc_html__( 'Show Arrows', 'azo' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'yes',
				'label_on'     => esc_html__( 'No', 'azo' ),
				'label_off'    => esc_html__( 'Yes', 'azo' ),
				'return_value' => 'yes',
			)
		);

		$this->add_control(
			'dots',
			array(
				'label'        => esc_html__( 'Show Dots', 'azo' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'no',
				'label_on'     => esc_html__( 'No', 'azo' ),
				'label_off'    => esc_html__( 'Yes', 'azo' ),
				'return_value' => 'yes',
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Section query
	 */
	private function section_query() {
		$this->start_controls_section(
			'product_query',
			array(
				'label' => esc_html__( 'Query', 'azo' ),
			)
		);

		$this->add_control(
			'cat_ids',
			array(
				'label' => esc_html__( 'Categories', 'azo' ),
				'type'  => 'autocomplete',
				'query' => array(
					'type' => 'term',
					'name' => 'product_cat',
				),
			)
		);

		$this->add_control(
			'product_ids',
			array(
				'label' => esc_html__( 'Products', 'azo' ),
				'type'  => 'autocomplete',
				'query' => array(
					'type' => 'post_type',
					'name' => 'product',
				),
			)
		);

		$this->add_control(
			'ex_cat_ids',
			array(
				'label' => esc_html__( 'Exclude Categories', 'azo' ),
				'type'  => 'autocomplete',
				'query' => array(
					'type' => 'term',
					'name' => 'product_cat',
				),
			)
		);

		$this->add_control(
			'ex_product_ids',
			array(
				'label' => esc_html__( 'Exclude Products', 'azo' ),
				'type'  => 'autocomplete',
				'query' => array(
					'type' => 'post_type',
					'name' => 'product',
				),
			)
		);

		$this->add_control(
			'count',
			array(
				'label'   => esc_html__( 'Posts Per Page', 'azo' ),
				'type'    => Controls_Manager::NUMBER,
				'default' => 6,
				'min'     => 1,
				'max'     => 100,
				'step'    => 1,
			)
		);

		$this->add_control(
			'order_by',
			array(
				'label'   => esc_html__( 'Order By', 'azo' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'id',
				'options' => array(
					'id'   => esc_html__( 'ID', 'azo' ),
					'name' => esc_html__( 'Name', 'azo' ),
					'date' => esc_html__( 'Date', 'azo' ),
					'rand' => esc_html__( 'Random', 'azo' ),
				),
			)
		);

		$this->add_control(
			'order',
			array(
				'label'   => esc_html__( 'Order', 'azo' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'ASC',
				'options' => array(
					'ASC'  => esc_html__( 'ASC', 'azo' ),
					'DESC' => esc_html__( 'DESC', 'azo' ),
				),
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Product
	 */
	private function product() {
		$this->start_controls_section(
			'product',
			array(
				'label' => __( 'Heading Product', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'heading_color',
			array(
				'label'     => esc_html__( 'Color Heading', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#222222',
				'selectors' => array(
					'{{WRAPPER}} .title-product-slider' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'heading_typo',
				'selector' => '{{WRAPPER}} .title-product-slider',
			)
		);

		$this->end_controls_section();

		// Arrows.
		$this->start_controls_section(
			'product_arrow_style',
			array(
				'label'     => esc_html__( 'Arrows', 'azo' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => array(
					'arrow' => 'yes',
				),
			)
		);

		// TAB START Arrow.
		$this->start_controls_tabs( 'product_arrow_tabs' );

		// Normal.
		$this->start_controls_tab(
			'product_arrow_normal',
			array(
				'label' => __( 'Normal', 'azo' ),
			)
		);

		// Border Color.
		$this->add_control(
			'product_arrow_border_color',
			array(
				'label'     => __( 'Border Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .title-wrappe .arrows-widget .azo-arrow' => 'border-color: {{VALUE}};',
				),
			)
		);

		// Color.
		$this->add_control(
			'product_arrow_color',
			array(
				'label'     => __( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .title-wrappe .arrows-widget .azo-arrow:before' => 'color: {{VALUE}};',
				),
			)
		);

		// Bg Color.
		$this->add_control(
			'product_arrow_bg_hover_color',
			array(
				'label'     => __( 'Background Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .title-wrappe .arrows-widget .azo-arrow' => 'background-color: {{VALUE}};',
				),
			)
		);

		// END NORMAL.
		$this->end_controls_tab();

		// HOVER.
		$this->start_controls_tab(
			'product_arrow_hover',
			array(
				'label' => __( 'Hover', 'azo' ),
			)
		);

		// Border Hover Color.
		$this->add_control(
			'product_arrow_border_hover_color',
			array(
				'label'     => __( 'Border Color Hover', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .title-wrappe .arrows-widget .azo-arrow:hover' => 'border-color: {{VALUE}};',
				),
			)
		);

		// Hover color.
		$this->add_control(
			'product_arrow_hover_color',
			array(
				'label'     => __( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .title-wrappe .arrows-widget .azo-arrow:hover:before' => 'color: {{VALUE}};',
				),
			)
		);

		// Bg Hover Color.
		$this->add_control(
			'product_arrow_bgcolor',
			array(
				'label'     => __( 'Background Hover Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .title-wrappe .arrows-widget .azo-arrow:hover' => 'background-color: {{VALUE}};',
				),
			)
		);

		// TAB END.
		$this->end_controls_tab();
		$this->end_controls_tabs();

		$this->end_controls_section();

		// Background color.
		$this->start_controls_section(
			'style_product_info',
			array(
				'label' => esc_html__( 'Style Product', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			array(
				'name'        => 'border',
				'placeholder' => '1px',
				'default'     => '1px',
				'selector'    => '{{WRAPPER}} .js-products-slider1',
				'condition'   => array(
					'class_slider' => '1',
				),
			)
		);

		$this->add_control(
			'border_radius',
			array(
				'label'      => __( 'Border Radius', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px' ),
				'selectors'  => array(
					'{{WRAPPER}} .js-products-slider1' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
				'condition'  => array(
					'class_slider' => '1',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			array(
				'name'        => 'border_slide2',
				'placeholder' => '1px',
				'default'     => '1px',
				'selector'    => '{{WRAPPER}} .js-products-slider2',
				'condition'   => array(
					'class_slider' => '2',
				),
			)
		);

		$this->add_control(
			'border_radius_slide2',
			array(
				'label'      => __( 'Border Radius', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px' ),
				'selectors'  => array(
					'{{WRAPPER}} .js-product-slider2' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
				'condition'  => array(
					'class_slider' => '2',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			array(
				'name'        => 'border_slide3',
				'placeholder' => '1px',
				'default'     => '1px',
				'selector'    => '{{WRAPPER}} .js-product-slider3',
				'condition'   => array(
					'class_slider' => '3',
				),
			)
		);

		$this->add_control(
			'border_radius_slide3',
			array(
				'label'      => __( 'Border Radius', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px' ),
				'selectors'  => array(
					'{{WRAPPER}} .js-product-slider3' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
				'condition'  => array(
					'class_slider' => '3',
				),
			)
		);

		$this->add_control(
			'product_info_bg_color',
			array(
				'label'     => __( 'Background Color Info Product', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-style' => 'background-color: {{VALUE}};',
				),
			)
		);

		// Gap Product Spacing.
		$this->add_responsive_control(
			'product_gap_spacing',
			array(
				'label'           => __( 'Spacing Product', 'azo' ),
				'type'            => Controls_Manager::SLIDER,
				'range'           => array(
					'px' => array(
						'max' => 200,
					),
				),
				'devices'         => array(
					'desktop',
					'tablet',
					'mobile',
				),
				'desktop_default' => array(
					'size' => 0,
					'unit' => 'px',
				),
				'tablet_default'  => array(
					'size' => 0,
					'unit' => 'px',
				),
				'mobile_default'  => array(
					'size' => 0,
					'unit' => 'px',
				),
				'selectors'       => array(
					'{{WRAPPER}} .slick-slide' => 'margin: 0 {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .slick-list'  => 'margin-left: -{{SIZE}}{{UNIT}}; margin-right: -{{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->end_controls_section();

		// Category.
		$this->start_controls_section(
			'product_category',
			array(
				'label' => esc_html__( 'Category', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'product_style_cat_typo',
				'label'    => esc_html__( 'Typography', 'azo' ),
				'selector' => '{{WRAPPER}} .azo-product-cats a',
			)
		);

		// TAB START.
		$this->start_controls_tabs( 'product_style_cat_tabs' );

		// Normal.
		$this->start_controls_tab(
			'product_style_cat_normal',
			array(
				'label' => __( 'Normal', 'azo' ),
			)
		);

		// Color.
		$this->add_control(
			'product_style_cat_color',
			array(
				'label'     => __( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-product-cats a' => 'color: {{VALUE}};',
				),
			)
		);

		// END NORMAL.
		$this->end_controls_tab();

		// HOVER.
		$this->start_controls_tab(
			'product_style_cat_hover',
			array(
				'label' => __( 'Hover', 'azo' ),
			)
		);

		// Hover color.
		$this->add_control(
			'product_style_cat_hover_color',
			array(
				'label'     => __( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-product-cats:hover a' => 'color: {{VALUE}};',
				),
			)
		);

		// TAB END.
		$this->end_controls_tab();
		$this->end_controls_tabs();

		$this->end_controls_section();

		// Title Product.
		$this->start_controls_section(
			'product_tab_title',
			array(
				'label' => esc_html__( 'Title Product', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'product_title',
			array(
				'label'     => __( 'Title', 'azo' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'product_style_title_typo',
				'label'    => esc_html__( 'Typography', 'azo' ),
				'selector' => '{{WRAPPER}} .woocommerce-loop-product__title',
			)
		);

		// Title Spacing.
		$this->add_responsive_control(
			'title_spacing',
			array(
				'label'           => __( 'Spacing', 'azo' ),
				'type'            => Controls_Manager::SLIDER,
				'range'           => array(
					'px' => array(
						'max' => 200,
					),
				),
				'devices'         => array(
					'desktop',
					'tablet',
					'mobile',
				),
				'desktop_default' => array(
					'size' => 0,
					'unit' => 'px',
				),
				'tablet_default'  => array(
					'size' => 0,
					'unit' => 'px',
				),
				'mobile_default'  => array(
					'size' => 0,
					'unit' => 'px',
				),
				'selectors'       => array(
					'{{WRAPPER}} .woocommerce-loop-product__title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				),
			)
		);

		// TAB START.
		$this->start_controls_tabs( 'product_style_tabs' );

		// Normal.
		$this->start_controls_tab(
			'product_style_normal',
			array(
				'label' => __( 'Normal', 'azo' ),
			)
		);

		// Color.
		$this->add_control(
			'product_style_title_color',
			array(
				'label'     => __( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .woocommerce-loop-product__title' => 'color: {{VALUE}};',
				),
			)
		);

		// END NORMAL.
		$this->end_controls_tab();

		// HOVER.
		$this->start_controls_tab(
			'product_style_hover',
			array(
				'label' => __( 'Hover', 'azo' ),
			)
		);

		// Hover color.
		$this->add_control(
			'product_style_color',
			array(
				'label'     => __( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} a:hover .woocommerce-loop-product__title' => 'color: {{VALUE}};',
				),
			)
		);

		// TAB END.
		$this->end_controls_tab();
		$this->end_controls_tabs();

		$this->end_controls_section();

		// Price Product.
		$this->start_controls_section(
			'product_tab_price',
			array(
				'label' => esc_html__( 'Price Product', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		// Price.
		$this->add_control(
			'product_price',
			array(
				'label'     => __( 'Price', 'azo' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		// Color Price.
		$this->add_control(
			'product_price_color',
			array(
				'label'     => __( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .price ins span, {{WRAPPER}} .price span' => 'color: {{VALUE}};',
				),
			)
		);

		// Price Typography.
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'product_price_typo',
				'label'    => esc_html__( 'Typography', 'azo' ),
				'selector' => '{{WRAPPER}} .price ins span, {{WRAPPER}} .price span',
			)
		);

		// Regular Price.
		$this->add_control(
			'product_regular_price',
			array(
				'label'     => __( 'Sale Price', 'azo' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		// Color Regular Price.
		$this->add_control(
			'product_regular_price_color',
			array(
				'label'     => __( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .price del span, {{WRAPPER}} .price del' => 'color: {{VALUE}};',
				),
			)
		);

		// Regular Price Typography.
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'product_regular_price_typo',
				'label'    => esc_html__( 'Typography', 'azo' ),
				'selector' => '{{WRAPPER}} .price del span',
			)
		);

		$this->end_controls_section();

		// Button Product.
		$this->start_controls_section(
			'product_tab_button',
			array(
				'label' => esc_html__( 'Button Product', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		// Style Button.
		$this->add_control(
			'product_button',
			array(
				'label'     => __( 'Button', 'azo' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		// Button Typography.
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'product_button_typo',
				'label'    => esc_html__( 'Typography', 'azo' ),
				'selector' => '{{WRAPPER}} .azo-add-to-cart-btn',
			)
		);

		// TAB START.
		$this->start_controls_tabs( 'atcbutton_style_tabs' );

		// Normal.
		$this->start_controls_tab(
			'atcbutton_style_normal',
			array(
				'label' => __( 'Normal', 'azo' ),
			)
		);

		// Color Button.
		$this->add_control(
			'atcbutton_button_color',
			array(
				'label'     => __( 'Color Button', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-add-to-cart-btn' => 'color: {{VALUE}};',
				),
			)
		);

		// Background Color Button.
		$this->add_control(
			'atcbutton_button_bg_color',
			array(
				'label'     => __( 'Background Color Button', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-add-to-cart-btn' => 'background-color: {{VALUE}};',
				),
			)
		);

		// END NORMAL.
		$this->end_controls_tab();

		// HOVER.
		$this->start_controls_tab(
			'atcbutton_style_hover',
			array(
				'label' => __( 'Hover', 'azo' ),
			)
		);

		// Color Hover Button.
		$this->add_control(
			'atcbutton_button_hover_color',
			array(
				'label'     => __( 'Color Hover Button', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-add-to-cart-btn:hover' => 'color: {{VALUE}};',
				),
			)
		);

		// Background Color Button.
		$this->add_control(
			'atcbutton_button_bg_hover_color',
			array(
				'label'     => __( 'Background Hover Color Button', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-add-to-cart-btn:hover' => 'background-color: {{VALUE}};',
				),
			)
		);

		// TAB END.
		$this->end_controls_tab();
		$this->end_controls_tabs();

		$this->end_controls_section();

		// Style Dot Slider.
		$this->start_controls_section(
			'product_rate',
			array(
				'label' => esc_html__( 'Rate', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		// Color Rate.
		$this->add_control(
			'product_rate_color',
			array(
				'label'     => __( 'Color Rate', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .star-rating > span:before' => 'color: {{VALUE}};',
				),
			)
		);

		$this->end_controls_section();

		// Style Stock Bar.
		$this->start_controls_section(
			'product_stockbar_style',
			array(
				'label'     => esc_html__( 'Stock Bar', 'azo' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => array(
					'stock_bar' => 'yes',
				),
			)
		);

		// Color Stock Bar.
		$this->add_control(
			'bg_status_stockbar_color',
			array(
				'label'     => __( 'Background Color Status Stock Bar', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .loop-product-stock .status-bar' => 'background-color: {{VALUE}};',
				),
			)
		);

		// Color.
		$this->add_control(
			'bg_stockbar_color',
			array(
				'label'     => __( 'Background Color Stock Bar', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .loop-product-stock .sold-bar' => 'background-color: {{VALUE}};',
				),
			)
		);

		// Stock Bar Typography.
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'stockbar_typo',
				'label'    => esc_html__( 'Typography Sold', 'azo' ),
				'selector' => '{{WRAPPER}} .product-stock-status .label',
			)
		);

		// Color.
		$this->add_control(
			'stockbar_color',
			array(
				'label'     => __( 'Color Stock Bar', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .product-stock-status .sold .label, .product-stock-status .available .label' => 'color: {{VALUE}};',
				),
			)
		);

		$this->end_controls_section();

		// Style Dot Slider.
		$this->start_controls_section(
			'dot_slider',
			array(
				'label'     => esc_html__( 'Dot Slide', 'azo' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => array(
					'dots' => array(
						'yes',
					),
				),
			)
		);

		// Color Dot Slider.
		$this->add_control(
			'dot_slider_color',
			array(
				'label'     => __( 'Color Dot Slider', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .product-slider-dots .dots-bullet:after' => 'background-color: {{VALUE}};',
				),
			)
		);

		// Border Color Dot Slider.
		$this->add_control(
			'border_dot_slider_color',
			array(
				'label'     => __( 'Border Color Dot Slider', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .product-slider-dots .dots-bullet:after' => 'border-color: {{VALUE}};',
				),
			)
		);

		// Color Dot Slider.
		$this->add_control(
			'dot_active_slider_color',
			array(
				'label'     => __( 'Color Active Dot Slider', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .product-slider-dots .slick-active .dots-bullet:after' => 'background-color: {{VALUE}};',
				),
			)
		);

		$this->add_control(
			'border_dot_active_slider_color',
			array(
				'label'     => __( 'Border Color Active Dot Slider', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .product-slider-dots .slick-active .dots-bullet:after' => 'border-color: {{VALUE}};',
				),
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Render
	 */
	public function render() {
		$settings = $this->get_settings_for_display();

		$paged = is_front_page() ? get_query_var( 'page' ) : get_query_var( 'paged' );
		$paged = $paged ? $paged : 1;
		$args  = array(
			'post_type'      => 'product',
			'post_status'    => 'publish',
			'posts_per_page' => $settings['count'],
			'orderby'        => $settings['order_by'],
			'order'          => $settings['order'],
			'paged'          => $paged,
		);

		// Category ids.
		$cat_ids    = $settings['cat_ids'];
		$ex_cat_ids = $settings['ex_cat_ids'];

		// Product ids.
		$product_ids    = $settings['product_ids'];
		$ex_product_ids = $settings['ex_product_ids'];

		if ( ! empty( $cat_ids ) ) {
			$args['tax_query'] = array( //phpcs:ignore
				array(
					'taxonomy' => 'product_cat',
					'field'    => 'term_id',
					'operator' => 'IN',
					'terms'    => $cat_ids,
				),
			);
		}

		if ( ! empty( $product_ids ) ) {
			$args['post__in'] = $product_ids;
		}

		if ( ! empty( $ex_product_ids ) ) {
			$args['post__not_in'] = $ex_product_ids;
		}

		$products = new \WP_Query( $args );
		if ( ! $products->have_posts() ) {
			return;
		}

		if ( '' === $settings['stock_bar'] ) {
			remove_action( 'woocommerce_shop_loop_item_title', 'azo_product_stock_bar', 10 );
		} else {
			add_action( 'woocommerce_shop_loop_item_title', 'azo_product_stock_bar', 10 );
		}
		?>
		<div class="azo-products-slider">
			<div class="product-js-products-slider woocommerce" data-show="<?php echo esc_attr( $settings['slides_to_show'] ); ?>" data-show-tablet="<?php echo esc_attr( $settings['slides_to_show_tablet'] ); ?>" data-show-mobile="<?php echo esc_attr( $settings['slides_to_show_mobile'] ); ?>" data-rows="<?php echo esc_attr( $settings['rows'] ); ?>">
				<div class="title-wrappe <?php echo esc_attr( $settings['style_arrow'] ); ?>">
					<h2 class="title-product-slider">
						<?php echo esc_html( $settings['title'] ); ?>
					</h2>

					<?php if ( 'yes' === $settings['arrow'] ) { ?>
						<div class="arrows-widget">
							<button class="azo-arrow arrows-prev<?php echo esc_attr( $settings['class_slider'] ); ?>"></button>
							<button class="azo-arrow arrows-next<?php echo esc_attr( $settings['class_slider'] ); ?>"></button>
						</div>
					<?php } ?>
				</div>

				<div class="products js-products-slider<?php echo esc_attr( $settings['class_slider'] ); ?>">
					<?php
					while ( $products->have_posts() ) {
						$products->the_post();
						get_template_part( 'template-parts/content', 'product' );
					}

					wp_reset_postdata();
					?>
				</div>

				<?php if ( 'yes' === $settings['dots'] ) { ?>
					<div class="product-slider-dots"></div>
				<?php } ?>
			</div>
		</div>
		<?php
	}
}
Plugin::instance()->widgets_manager->register_widget_type( new Azo_Products_Slider() );
