<?php
/**
 * Elementor Product Images Widget
 *
 * @package azo
 */

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

if ( ! defined( 'WC_PLUGIN_FILE' ) ) {
	return;
}

/**
 * Class for azo elementor product images widget.
 */
class Azo_Product_Filter extends \Elementor\Widget_Base {
	/**
	 * Category
	 */
	public function get_categories() {
		return array( 'azo-theme' );
	}

	/**
	 * Name
	 */
	public function get_name() {
		return 'azo-product-filter';
	}

	/**
	 * Gets the title.
	 */
	public function get_title() {
		return __( 'Product Filter', 'azo' );
	}

	/**
	 * Gets the icon.
	 */
	public function get_icon() {
		return 'eicon-tabs';
	}

	/**
	 * Retrieve the list of scripts the widget depended on
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return array( 'azo-product-filter' );
	}

	/**
	 * Controls
	 */
	protected function register_controls() { //phpcs:ignore
		$this->start_controls_section(
			'section_product_filter',
			array(
				'label' => esc_html__( 'Product Filter', 'azo' ),
			)
		);

		$this->add_control(
			'gap',
			array(
				'label'   => esc_html__( 'Gap', 'azo' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'gap-default' => esc_html__( 'Gap Default', 'azo' ),
					'gap-min'     => esc_html__( 'Gap Min', 'azo' ),
				),
				'default' => 'gap-default',
			)
		);

		$this->add_control(
			'tab',
			array(
				'label'    => esc_html__( 'Tab', 'azo' ),
				'type'     => Controls_Manager::SELECT2,
				'multiple' => true,
				'options'  => array(
					'bestseller' => esc_html__( 'Best Selling', 'azo' ),
					'onsalet'    => esc_html__( 'On Sale', 'azo' ),
					'trending'   => esc_html__( 'Trending', 'azo' ),
					'recent'     => esc_html__( 'Recent', 'azo' ),
					'popular'    => esc_html__( 'Popular', 'azo' ),
				),
				'default'  => array( 'bestseller' ),
			)
		);

		$this->add_control(
			'number',
			array(
				'label'   => esc_html__( 'Products Per Page', 'azo' ),
				'type'    => Controls_Manager::NUMBER,
				'default' => 4,
			)
		);

		$this->add_control(
			'rows',
			array(
				'label'   => esc_html__( 'Rows', 'azo' ),
				'type'    => Controls_Manager::NUMBER,
				'default' => 1,
			)
		);

		$this->add_control(
			'autoplay',
			array(
				'label'     => esc_html__( 'Autoplay', 'azo' ),
				'type'      => Controls_Manager::SWITCHER,
				'label_on'  => esc_html__( 'No', 'azo' ),
				'label_off' => esc_html__( 'Yes', 'azo' ),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_filter',
			array(
				'label' => esc_html__( 'Filter Tab', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'text_align',
			array(
				'label'     => esc_html__( 'Alignment', 'azo' ),
				'type'      => Controls_Manager::CHOOSE,
				'options'   => array(
					'flex-start' => array(
						'title' => esc_html__( 'Left', 'azo' ),
						'icon'  => 'fa fa-align-left',
					),
					'center'     => array(
						'title' => esc_html__( 'Center', 'azo' ),
						'icon'  => 'fa fa-align-center',
					),
					'flex-end'   => array(
						'title' => esc_html__( 'Right', 'azo' ),
						'icon'  => 'fa fa-align-right',
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .product-filter-tab ul' => 'justify-content:{{VALUE}};',
				),
			)
		);

		$this->add_control(
			'color_filter',
			array(
				'label'     => __( 'Color Filter', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#757575',
				'selectors' => array(
					'{{WRAPPER}} .product-filter-tab .tab-item' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'color_filter_active',
			array(
				'label'     => __( 'Color Activer Filter', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffb700',
				'selectors' => array(
					'{{WRAPPER}} .product-filter-tab .tab-item.active' => 'color: {{VALUE}}',
					'{{WRAPPER}} .product-filter-tab .tab-item:hover' => 'color: {{VALUE}}',
					'{{WRAPPER}} .product-filter-tab .tab-item:before' => 'background-color: {{VALUE}}',
					'{{WRAPPER}} .product-filter-tab .tab-item.active:before' => 'background-color: {{VALUE}}',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'filter_typography',
				'selector' => '{{WRAPPER}} .product-filter-tab .tab-item',
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_content',
			array(
				'label' => esc_html__( 'Content Tab', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'color_title',
			array(
				'label'     => __( 'Color Title', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#333333',
				'selectors' => array(
					'{{WRAPPER}} .woocommerce-loop-product__title a' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'color_hover_title',
			array(
				'label'     => __( 'Color Hover Title', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffb700',
				'selectors' => array(
					'{{WRAPPER}} .woocommerce-loop-product__title:hover a' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'title_typography',
				'selector' => '{{WRAPPER}} .woocommerce-loop-product__title a',
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_dots',
			array(
				'label'     => esc_html__( 'Dots', 'azo' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => array(
					'dots' => 'yes',
				),
			)
		);

		$this->add_control(
			'dots_color',
			array(
				'label'     => esc_html__( 'Dots Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#d2d2d2',
				'selectors' => array(
					'{{WRAPPER}} .product-slider-dots .dots-bullet:after' => 'background-color: {{VALUE}}',
				),
				'condition' => array(
					'dots' => 'yes',
				),
			)
		);

		$this->add_control(
			'dots_hover_color',
			array(
				'label'     => esc_html__( 'Dots Active Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#2a5cb8',
				'selectors' => array(
					'{{WRAPPER}} .product-slider-dots .slick-active .dots-bullet:after' => 'background-color: {{VALUE}};',
				),
				'condition' => array(
					'dots' => 'yes',
				),
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Render
	 */
	public function render() {
		$settings  = $this->get_settings_for_display();
		$tabname   = $this->azo_get_tab_name();
		$classes   = 'tab-item ';
		$style_gap = $settings['gap'];

		if ( ! $tabname ) {
			return;
		}
		foreach ( $tabname as $slug => $name ) {
			$slugs[] = $slug;
		}
		$tabactive = $slugs[0];
		?>
		<div class="azo-product-filter <?php echo esc_attr( $style_gap ); ?>">
			<div class="product-filter--wrapper">
				<div class="product-filter-tab">
					<ul>
					<?php
					foreach ( $tabname as $setting => $name ) {
						$active = ( $setting === $tabactive ) ? ' active' : '';
						?>
							<li class="<?php echo esc_attr( $classes . $setting . $active ); ?>" data-content="<?php echo esc_attr( $setting ); ?>">
								<?php echo esc_html( $name ); ?>
							</li>
					<?php } ?>

					</ul>
				</div>

				<div class="product-filter-content woocommerce" data-autoplay="<?php echo esc_attr( $settings['autoplay'] ); ?>" data-rows="<?php echo esc_attr( $settings['rows'] ); ?>">
					<div class="preloader-filter"></div>

					<?php
					foreach ( $tabname as $setting => $name ) {
						if ( 'bestseller' === $setting ) {
							azo_product_bestseller( $settings['number'] );
						}

						if ( 'onsalet' === $setting ) {
							azo_product_on_sale( $settings['number'] );
						}

						if ( 'trending' === $setting ) {
							azo_product_trending( $settings['number'] );
						}

						if ( 'popular' === $setting ) {
							azo_product_popular( $settings['number'] );
						}

						if ( 'recent' === $setting ) {
							azo_product_recent( $settings['number'] );
						}
					}
					?>
				</div>
			</div>
		</div>
		<?php
		wp_reset_postdata();
	}

	/**
	 * Azo get tab name
	 */
	private function azo_get_tab_name() {
		$settings = $this->get_settings_for_display();
		$tabs     = $settings['tab'];
		foreach ( $tabs as $tab ) {

			if ( 'bestseller' === $tab ) {
				$tabname[ $tab ] = 'Best Selling';
			}
			if ( 'onsalet' === $tab ) {
				$tabname[ $tab ] = 'On Sale';
			}
			if ( 'trending' === $tab ) {
				$tabname[ $tab ] = 'Trending';
			}

			if ( 'recent' === $tab ) {
				$tabname[ $tab ] = 'Recent';
			}

			if ( 'popular' === $tab ) {
				$tabname[ $tab ] = 'Popular';
			}
		}
		return $tabname;
	}
}
Plugin::instance()->widgets_manager->register_widget_type( new Azo_Product_Filter() );
