<?php
/**
 * Elementor Product Filter By Category Widget
 *
 * @package azo
 */

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

if ( ! defined( 'WC_PLUGIN_FILE' ) ) {
	return;
}

/**
 * Class for azo elementor product images widget.
 */
class Azo_Product_Filter_By_Category extends \Elementor\Widget_Base {
	/**
	 * Category
	 */
	public function get_categories() {
		return array( 'azo-theme' );
	}

	/**
	 * Name
	 */
	public function get_name() {
		return 'azo-product-filter-by-category';
	}

	/**
	 * Gets the title.
	 */
	public function get_title() {
		return __( 'Product Filter By Category', 'azo' );
	}

	/**
	 * Gets the icon.
	 */
	public function get_icon() {
		return 'eicon-tabs';
	}

	/**
	 * Retrieve the list of scripts the widget depended on
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return array( 'azo-product-filter-by-category' );
	}

	/**
	 * Controls
	 */
	protected function register_controls() { //phpcs:ignore
		$this->start_controls_section(
			'section_filter_product',
			array(
				'label' => esc_html__( 'Product Filter By Category', 'azo' ),
			)
		);

		$this->add_control(
			'gap',
			array(
				'label'   => esc_html__( 'Gap', 'azo' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'gap-default' => esc_html__( 'Gap Default', 'azo' ),
					'gap-min'     => esc_html__( 'Gap Min', 'azo' ),
				),
				'default' => 'gap-default',
			)
		);

		$this->add_control(
			'select_by',
			array(
				'label'    => esc_html__( 'Select Category', 'azo' ),
				'type'     => Controls_Manager::SELECT2,
				'multiple' => true,
				'options'  => $this->term_id(),
			)
		);

		$this->add_control(
			'number',
			array(
				'label'   => esc_html__( 'Products Per Page', 'azo' ),
				'type'    => Controls_Manager::NUMBER,
				'default' => 4,
			)
		);

		// Slides to show.
		$this->add_responsive_control(
			'slides_to_show',
			array(
				'label'                 => esc_html__( 'Slides to show', 'azo' ),
				'type'                  => Controls_Manager::NUMBER,
				'min'                   => 1,
				'max'                   => 5,
				'step'                  => 1,
				'devices'               => array( 'desktop', 'tablet', 'mobile' ),
				'slides_to_show_mobile' => array(
					'default' => 1,
				),
				'slides_to_show_tablet' => array(
					'default' => 2,
				),
				'slides_to_show'        => array(
					'default' => 2,
				),
			)
		);

		$this->add_control(
			'rows',
			array(
				'label'   => esc_html__( 'Rows', 'azo' ),
				'type'    => Controls_Manager::NUMBER,
				'default' => 1,
			)
		);

		$this->add_control(
			'autoplay',
			array(
				'label'     => esc_html__( 'Autoplay', 'azo' ),
				'type'      => Controls_Manager::SWITCHER,
				'label_on'  => esc_html__( 'No', 'azo' ),
				'label_off' => esc_html__( 'Yes', 'azo' ),
			)
		);

		$this->add_control(
			'arrow',
			array(
				'label'        => esc_html__( 'Show Arrow', 'azo' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'yes',
				'label_on'     => esc_html__( 'No', 'azo' ),
				'label_off'    => esc_html__( 'Yes', 'azo' ),
				'return_value' => 'yes',
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_filter',
			array(
				'label' => esc_html__( 'Filter Tab', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'text_align',
			array(
				'label'     => esc_html__( 'Alignment', 'azo' ),
				'type'      => Controls_Manager::CHOOSE,
				'options'   => array(
					'flex-start' => array(
						'title' => esc_html__( 'Left', 'azo' ),
						'icon'  => 'fa fa-align-left',
					),
					'center'     => array(
						'title' => esc_html__( 'Center', 'azo' ),
						'icon'  => 'fa fa-align-center',
					),
					'flex-end'   => array(
						'title' => esc_html__( 'Right', 'azo' ),
						'icon'  => 'fa fa-align-right',
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .product-filter-tab ul' => 'justify-content:{{VALUE}};',
				),
			)
		);

		$this->add_control(
			'color_filter',
			array(
				'label'     => __( 'Color Filter', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#757575',
				'selectors' => array(
					'{{WRAPPER}} .product-filter-tab .tab-item-filter' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'color_filter_active',
			array(
				'label'     => __( 'Color Activer Filter', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffb700',
				'selectors' => array(
					'{{WRAPPER}} .product-filter-tab .tab-item-filter.active' => 'color: {{VALUE}}',
					'{{WRAPPER}} .product-filter-tab .tab-item-filter:hover' => 'color: {{VALUE}}',
					'{{WRAPPER}} .product-filter-tab .tab-item-filter:before' => 'background-color: {{VALUE}}',
					'{{WRAPPER}} .product-filter-tab .tab-item-filter.active:before' => 'background-color: {{VALUE}}',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'filter_typography',
				'selector' => '{{WRAPPER}} .product-filter-tab .tab-item-filter',
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_content',
			array(
				'label' => esc_html__( 'Content Tab', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'color_title',
			array(
				'label'     => __( 'Color Title', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#333333',
				'selectors' => array(
					'{{WRAPPER}} .woocommerce-loop-product__title a' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'color_hover_title',
			array(
				'label'     => __( 'Color Hover Title', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffb700',
				'selectors' => array(
					'{{WRAPPER}} .woocommerce-loop-product__title:hover a' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'title_typography',
				'selector' => '{{WRAPPER}} .woocommerce-loop-product__title a',
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_arrow',
			array(
				'label'     => esc_html__( 'Arrow', 'azo' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => array(
					'arrow' => 'yes',
				),
			)
		);

		$this->add_control(
			'arrow_color',
			array(
				'label'     => esc_html__( 'Arrows Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#e5e5e5',
				'selectors' => array(
					'{{WRAPPER}} .product-filter-category-content .arrows-widget .azo-arrow' => 'border-color: {{VALUE}}',
					'{{WRAPPER}} .product-filter-category-content .arrows-widget .azo-arrow:before' => 'color: {{VALUE}}',
				),
				'condition' => array(
					'arrow' => 'yes',
				),
			)
		);

		$this->add_control(
			'arrow_hover_color',
			array(
				'label'     => esc_html__( 'Arrows Hover Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffb700',
				'selectors' => array(
					'{{WRAPPER}} .product-filter-category-content .arrows-widget .azo-arrow:hover' => 'border-color: {{VALUE}};background-color: {{VALUE}};',
					'{{WRAPPER}} .product-filter-category-content .arrows-widget .azo-arrow:focus' => 'border-color: {{VALUE}};background-color: {{VALUE}};',
				),
				'condition' => array(
					'arrow' => 'yes',
				),
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Render
	 */
	public function render() {
		$settings  = $this->get_settings_for_display();
		$cat_id    = $settings['select_by'];
		$classes   = 'tab-item-filter ';
		$style_gap = $settings['gap'];

		?>
		<div class="azo-product-filter-by-category <?php echo esc_attr( $style_gap ); ?>">
			<div class="product-cat-filter--wrapper">
				<div class="product-filter-tab">
					<ul>
						<?php
						foreach ( $cat_id as $key => $value ) {
							$term = get_term( (int) $value, 'product_cat' );
							if ( null !== $term ) {
								$term_name = $term->name;
								$slug      = $term->slug;

								$active = ( 0 === $key ) ? ' active' : '';
							} else {
								return;
							}

							?>
							<li class="<?php echo esc_attr( $classes . $slug . $active ); ?>" data-content="<?php echo esc_attr( $slug ); ?>">
								<?php echo esc_html( $term_name ); ?>
							</li>
						<?php } ?>
					</ul>
				</div>

				<div class="product-filter-category-content woocommerce" data-show="<?php echo esc_attr( $settings['slides_to_show'] ); ?>" data-show-tablet="<?php echo esc_attr( $settings['slides_to_show_tablet'] ); ?>" data-show-mobile="<?php echo esc_attr( $settings['slides_to_show_mobile'] ); ?>" data-rows="<?php echo esc_attr( $settings['rows'] ); ?>" data-autoplay="<?php echo esc_attr( $settings['autoplay'] ); ?>">
					<?php
					foreach ( $cat_id as $key => $value ) {
						$term      = get_term( (int) $value, 'product_cat' );
						$term_name = $term->name;
						$slug      = $term->slug;
						$args      = array(
							'post_type'           => 'product',
							'posts_per_page'      => $settings['number'],
							'ignore_sticky_posts' => 1,
							'tax_query'           => array( //phpcs:ignore
								array(
									'taxonomy' => 'product_cat',
									'field'    => 'slug',
									'terms'    => $slug,
								),
							),
						);

						$products = new \WP_Query( $args );
						?>
						<div class="products ht-grid data-content-filter tab-<?php echo esc_attr( $slug ); ?> js-products-slider" data-content='<?php echo esc_attr( $slug ); ?>'>
							<?php
							while ( $products->have_posts() ) {
								$products->the_post();
								get_template_part( 'template-parts/content', 'product' );
							}
							?>
						</div>
						<?php
					}
					?>
				</div>
			</div>
		</div>
		<?php
		wp_reset_postdata();
	}

	/**
	 * Get category
	 */
	public function term_id() {
		$terms = get_terms(
			'product_cat',
			array(
				'hide_empty' => false,
			)
		);

		return wp_list_pluck( $terms, 'name', 'term_id' );
	}
}
Plugin::instance()->widgets_manager->register_widget_type( new Azo_Product_Filter_By_Category() );
