<?php
/**
 * Azo Landing Image
 *
 * @package azo
 */

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	return;
}

/**
 * Azo landing image widget.
 *
 * Azo widget that displays an landing image for landing page.
 *
 * @since 1.0.0
 */
class Azo_Landing_Image extends Widget_Base {
	/**
	 * Get widget name.
	 *
	 * Retrieve landing image widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'Azo_landing_image';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve landing image widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Landing Image', 'azo' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve landing image widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-image-rollover';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the icon widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return array( 'azo-theme' );
	}

	/**
	 * Register category box widget controls.
	 *
	 * Add different input fields to allow the user to change and customize the widget settings
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function _register_controls() { //phpcs:ignore
		$this->start_controls_section(
			'section_landing',
			array(
				'label' => esc_html__( 'Landing Image', 'azo' ),
			)
		);

		$this->add_control(
			'image',
			array(
				'label'   => esc_html__( 'Choose Image', 'azo' ),
				'type'    => Controls_Manager::MEDIA,
				'dynamic' => array(
					'active' => true,
				),
				'default' => array(
					'url' => Utils::get_placeholder_image_src(),
				),
			)
		);

		$this->add_control(
			'title_text',
			array(
				'label'       => esc_html__( 'Title Text', 'azo' ),
				'type'        => Controls_Manager::TEXT,
				'dynamic'     => array(
					'active' => true,
				),
				'default'     => esc_html__( 'This is the heading', 'azo' ),
				'placeholder' => esc_html__( 'Enter your text title', 'azo' ),
				'label_block' => true,
			)
		);

		$this->add_control(
			'link',
			array(
				'label'       => esc_html__( 'Link to', 'azo' ),
				'type'        => Controls_Manager::URL,
				'dynamic'     => array(
					'active' => true,
				),
				'placeholder' => esc_html__( 'https://your-link.com', 'azo' ),
				'separator'   => 'before',
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_content',
			array(
				'label' => esc_html__( 'Content', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'title_style',
			array(
				'label'     => esc_html__( 'Title', 'azo' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'title_typography',
				'selector' => '{{WRAPPER}} .azo-landing-image-title',
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Render landing image widget output on the front end.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$image     = $settings['image'];
		$image_url = $image['url'];
		$image_alt = azo_img_alt( $image['id'], esc_attr__( 'Landing Image', 'azo' ) );

		$href = '#';
		$attr = '';

		if ( ! empty( $settings['link']['url'] ) ) {
			$href = $settings['link']['url'];

			if ( 'on' === $settings['link']['is_external'] ) {
				$attr .= ' target="_blank"';
			}

			if ( 'on' === $settings['link']['nofollow'] ) {
				$attr .= ' rel="nofollow"';
			}
		}
		?>
		<div class="azo-landing-image">
			<div class="card">
				<div class="card__image">
					<a href="<?php echo esc_url( $href ); ?>" <?php echo wp_kses( $attr, azo_get_allowed_html() ); ?> class="azo-landing-image-link">
						<span class="azo-landing-image-text"><?php echo esc_html__( 'View Demo', 'azo' ); ?></span>
					</a>

					<img src="<?php echo esc_url( $image_url ); ?>" alt="<?php esc_attr_e( 'image-landing', 'azo' ); ?>" />
				</div>
			</div>

			<div class="azo-landing-image-content">
				<h3 class="azo-landing-image-title">
					<a href="<?php echo esc_url( $href ); ?>" <?php echo wp_kses( $attr, azo_get_allowed_html() ); ?>><?php echo esc_html( $settings['title_text'] ); ?></a>
				</h3>
			</div>
		</div><!-- .azo-landing-image -->
		<?php
	}
}

Plugin::instance()->widgets_manager->register_widget_type( new Azo_Landing_Image() );
