<?php
/**
 * Azo instagram slider widget
 *
 * @package azo
 */

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Azo instagram slider widget.
 *
 * Azo widget that displays an image, a headline, a text and a button.
 *
 * @since 1.0.0
 */
class Azo_Instagram_Slider extends Widget_Base {
	/**
	 * Get widget name.
	 *
	 * Retrieve instagram slider widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'azo-instagram-slider';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve instagram box widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Azo Instagram Slider', 'azo' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve instagram box widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-carousel';
	}

	/**
	 * Retrieve the list of scripts the widget depended on
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return array( 'azo-instagram-slider' );
	}

	/**
	 * Get categories
	 */
	public function get_categories() {
		return array( 'azo-theme' );
	}

	/**
	 * Register instagram box widget controls.
	 *
	 * Add different input fields to allow the user to change and customize the widget settings
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() { //phpcs:ignore
		$this->start_controls_section(
			'section_instagram',
			array(
				'label' => esc_html__( 'instagram Slider', 'azo' ),
			)
		);

		$repeater = new Repeater();

		$repeater->add_control(
			'image-slider',
			array(
				'label'   => esc_html__( 'Choose Image', 'azo' ),
				'type'    => Controls_Manager::MEDIA,
				'dynamic' => array(
					'active' => true,
				),
				'default' => array(
					'url' => Utils::get_placeholder_image_src(),
				),
			)
		);

		$repeater->add_control(
			'selected_icon',
			array(
				'label'            => esc_html__( 'Icon', 'azo' ),
				'type'             => Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'default'          => array(
					'value' => 'fab fa-instagram',
				),
			)
		);

		$repeater->add_control(
			'link',
			array(
				'label'       => esc_html__( 'Link to', 'azo' ),
				'type'        => Controls_Manager::URL,
				'dynamic'     => array(
					'active' => true,
				),
				'placeholder' => esc_html__( 'https://your-link.com', 'azo' ),
			)
		);

		$this->add_control(
			'list-instagram-slider',
			array(
				'label'       => esc_html__( 'List instagram', 'azo' ),
				'type'        => Controls_Manager::REPEATER,
				'fields'      => $repeater->get_controls(),
				'title_filed' => '{{{ name }}}',
				'default'     => array(
					array(),
					array(),
					array(),
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_layout',
			array(
				'label' => esc_html__( 'Layout', 'azo' ),
			)
		);

		$this->add_responsive_control(
			'slides_to_show',
			array(
				'label'                 => esc_html__( 'Slides to show', 'azo' ),
				'type'                  => Controls_Manager::NUMBER,
				'min'                   => 1,
				'max'                   => 6,
				'step'                  => 1,
				'devices'               => array( 'desktop', 'tablet', 'mobile' ),
				'slides_to_show_mobile' => array( 'default' => 1 ),
				'slides_to_show_tablet' => array( 'default' => 2 ),
				'slides_to_show'        => array( 'default' => 3 ),
			)
		);

		$this->add_control(
			'dots',
			array(
				'label'        => esc_html__( 'Show Dots', 'azo' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'no',
				'label_on'     => esc_html__( 'No', 'azo' ),
				'label_off'    => esc_html__( 'Yes', 'azo' ),
				'return_value' => 'yes',
			)
		);

		$this->add_control(
			'arrows',
			array(
				'label'        => esc_html__( 'Show Arrows', 'azo' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'yes',
				'label_on'     => esc_html__( 'No', 'azo' ),
				'label_off'    => esc_html__( 'Yes', 'azo' ),
				'return_value' => 'yes',
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style-info',
			array(
				'label' => esc_html__( 'Info', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'br-image',
			array(
				'label'      => esc_html__( 'Border Radius', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .box-images-slider img' => 'border-radius:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				),
			)
		);

		$this->add_responsive_control(
			'icon_intasgram',
			array(
				'label'     => esc_html__( 'Size Icon Instagram', 'azo' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min' => 1,
						'max' => 300,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .box-icon .icon svg' => 'width: {{SIZE}}{{UNIT}}; height:auto;',
					'{{WRAPPER}} .box-icon .icon'     => 'font-size: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->add_control(
			'icon_intasgram_color',
			array(
				'label'     => esc_html__( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .box-icon .icon svg' => 'fill: {{VALUE}}',
					'{{WRAPPER}} .box-icon .icon'     => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'icon_intasgram_hover',
			array(
				'label'     => esc_html__( 'Hover Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-link:hover svg' => 'fill: {{VALUE}}',
					'{{WRAPPER}} .azo-link:hover .icon' => 'color: {{VALUE}}',
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_arrows',
			array(
				'label' => esc_html__( 'Arrows, Dots', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'dots_color',
			array(
				'label'     => esc_html__( 'Dots Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#222222',
				'selectors' => array(
					'{{WRAPPER}} .instagram-slider-dots .dots-bullet:after' => 'background-color: {{VALUE}}',
					'{{WRAPPER}} .instagram-slider-dots .slick-active .dots-bullet:after' => 'border-color: {{VALUE}}',
				),
				'separator' => 'before',
				'condition' => array(
					'dots' => 'yes',
				),
			)
		);

		$this->add_responsive_control(
			'dots_padding',
			array(
				'label'     => esc_html__( 'Margin top', 'azo' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min' => 0,
						'max' => 1000,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .instagram-slider-dots .slick-dots' => 'margin-top: {{SIZE}}{{UNIT}}',
				),
			)
		);

		$this->add_control(
			'arrows_color',
			array(
				'label'     => esc_html__( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#2f2f30',
				'selectors' => array(
					'{{WRAPPER}} .azo-slider-instagram-widget .arrows-widget .arrows-prev:before' => 'color:{{VALUE}}',
					'{{WRAPPER}} .azo-slider-instagram-widget .arrows-widget .arrows-next:before' => 'color:{{VALUE}}',
				),
			)
		);

		$this->add_control(
			'arrows_color_hover',
			array(
				'label'     => esc_html__( 'Color hover', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-slider-instagram-widget .arrows-widget .arrows-prev:hover:before' => 'color:{{VALUE}}',
					'{{WRAPPER}} .azo-slider-instagram-widget .arrows-widget .arrows-next:hover:before' => 'color:{{VALUE}}',
				),
			)
		);

		$this->end_controls_section();

	}

	/**
	 * Render
	 */
	public function render() {
		$settings = $this->get_settings_for_display();
		$dot      = $settings['dots'];
		?>
		<div class="azo-slider-instagram-widget">
			<div class="images-slider-wrapper" data-show="<?php echo esc_attr( $settings['slides_to_show'] ); ?>" data-show-tablet="<?php echo esc_attr( $settings['slides_to_show_tablet'] ); ?>" data-show-mobile="<?php echo esc_attr( $settings['slides_to_show_mobile'] ); ?>" >

				<?php if ( 'yes' === $settings['arrows'] ) { ?>
					<div class="arrows-widget">
						<button class="azo-arrow arrows-prev"></button>
						<button class="azo-arrow arrows-next"></button>
					</div>
				<?php } ?>

				<div class="js-instagram-slideshow list-instagram" data-dot="<?php echo esc_attr( $dot ); ?>">
					<?php foreach ( $settings['list-instagram-slider'] as $instagram_item ) { ?>
						<div class="azo-slider-item-instagram">
							<div class="box-images-slider">
								<a href="<?php echo esc_url( $instagram_item['link']['url'] ); ?>">
									<img src="<?php echo esc_url( $instagram_item['image-slider']['url'] ); ?>" alt="<?php echo esc_attr( $instagram_item['name'] ); ?>" class="slider-images">
								</a>
							</div>
							<?php
							if ( ! empty( $instagram_item['selected_icon']['value'] ) ) {
								?>
							<div class="box-icon">

								<?php if ( is_string( $instagram_item['selected_icon']['value'] ) ) { ?>
									<a href="<?php echo esc_url( $instagram_item['link']['url'] ); ?>" class="azo-link">
										<span class="icon">
											<i class="<?php echo esc_attr( $instagram_item['selected_icon']['value'] ); ?>"></i>
										</span>
									</a>
								<?php } else { ?>
									<a href="<?php echo esc_url( $instagram_item['link']['url'] ); ?>" class="azo-link">
										<span class="icon">
										<?php Icons_Manager::render_icon( $instagram_item['selected_icon'], '' ); ?>
										</span>
									</a>
									<?php
								}
								?>
							</div>
							<?php } ?>
						</div>						
					<?php } ?>
				</div>
				<div class="instagram-slider-dots"></div>
			</div>
		</div>
		<?php
	}
}

Plugin::instance()->widgets_manager->register_widget_type( new Azo_Instagram_Slider() );
