<?php
/**
 * Elementor Heading
 *
 * @package azo
 */

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Azo_Heading elementor posts widget.
 */
class Azo_Heading extends \Elementor\Widget_Base {

	/**
	 * Name
	 */
	public function get_name() {
		return 'azo-heading';
	}

	/**
	 * Title
	 */
	public function get_title() {
		return esc_html__( 'Heading Azo', 'azo' );
	}

	/**
	 * Icon
	 */
	public function get_icon() {
		return 'eicon-t-letter';
	}

	/**
	 * Category
	 */
	public function get_categories() {
		return array( 'azo-theme' );
	}

	/**
	 * Controls
	 */
	protected function _register_controls() { //phpcs:ignore
		$this->start_controls_section(
			'post_content',
			array(
				'label' => esc_html__( 'General', 'azo' ),
			)
		);

		// Title Heading.
		$this->add_control(
			'widget_title',
			array(
				'label'   => esc_html__( 'Title', 'azo' ),
				'type'    => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__( 'Best Sellers', 'azo' ),
			)
		);

		// Sub Title Heading.
		$this->add_control(
			'widget_sub_title',
			array(
				'label'   => esc_html__( 'Sub title', 'azo' ),
				'type'    => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__( 'All Products', 'azo' ),
			)
		);

		$this->add_control(
			'link',
			array(
				'label'     => __( 'Link', 'azo' ),
				'type'      => Controls_Manager::URL,
				'dynamic'   => array(
					'active' => true,
				),
				'default'   => array(
					'url' => '',
				),
				'separator' => 'before',
			)
		);

		// Alignment.
		$this->add_control(
			'alignment',
			array(
				'type'      => Controls_Manager::CHOOSE,
				'label'     => esc_html__( 'Alignment', 'azo' ),
				'options'   => array(
					'flex-start'    => array(
						'title' => esc_html__( 'Left', 'azo' ),
						'icon'  => 'fa fa-align-left',
					),
					'center'        => array(
						'title' => esc_html__( 'Center', 'azo' ),
						'icon'  => 'fa fa-align-center',
					),
					'flex-end'      => array(
						'title' => esc_html__( 'Right', 'azo' ),
						'icon'  => 'fa fa-align-right',
					),
					'space-between' => array(
						'title' => esc_html__( 'Space Between', 'azo' ),
						'icon'  => 'fa fa-align-justify',
					),
				),
				'default'   => '',
				'selectors' => array(
					'{{WRAPPER}} .azo-section-heading' => 'justify-content: {{VALUE}};',
				),
			)
		);

		$this->end_controls_section();

		/**
		 * Tab style heading
		 */
		$this->start_controls_section(
			'content_section',
			array(
				'label' => esc_html__( 'Style heading', 'azo' ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'tab_heading',
			array(
				'label' => __( 'Tab Heading', 'azo' ),
				'type'  => Controls_Manager::HEADING,
			)
		);

		// Color & typography title.
		$this->add_control(
			'heading_color',
			array(
				'label'     => esc_html__( 'Heading Color', 'azo' ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-section-heading h2' => 'color: {{VALUE}}',
				),
			)
		);

		// Typography title.
		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'heading_typography',
				'label'    => esc_html__( 'Typography Heading', 'azo' ),
				'selector' => '{{WRAPPER}} .azo-section-heading h2',
			)
		);

		// Title sub.
		$this->add_control(
			'tab_button',
			array(
				'label'     => __( 'Tab Button', 'azo' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_control(
			'title_button_color',
			array(
				'label'     => esc_html__( 'Title Button Color', 'azo' ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .subtitle a'         => 'color: {{VALUE}}',
					'{{WRAPPER}} .azo-icon svg' => 'fill: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'title_button_hover_color',
			array(
				'label'     => esc_html__( 'Title Button Hover Color', 'azo' ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .subtitle a:hover' => 'color: {{VALUE}}',
					'{{WRAPPER}} .subtitle a:hover .azo-icon svg' => 'fill: {{VALUE}}',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'button_typography',
				'label'    => esc_html__( 'Typography Title Button', 'azo' ),
				'selector' => '{{WRAPPER}} .subtitle',
			)
		);

		// Title Spacing.
		$this->add_responsive_control(
			'button_spacing',
			array(
				'label'           => __( 'Button Space Left', 'azo' ),
				'type'            => Controls_Manager::SLIDER,
				'range'           => array(
					'px' => array(
						'max' => 200,
					),
				),
				'devices'         => array(
					'desktop',
					'tablet',
					'mobile',
				),
				'desktop_default' => array(
					'size' => 29,
					'unit' => 'px',
				),
				'tablet_default'  => array(
					'size' => 0,
					'unit' => 'px',
				),
				'mobile_default'  => array(
					'size' => 0,
					'unit' => 'px',
				),
				'selectors'       => array(
					'{{WRAPPER}} .subtitle' => 'margin-left: {{SIZE}}{{UNIT}};',
				),
			)
		);

		// Tab Icon.
		$this->add_control(
			'icon_tab',
			array(
				'label'     => __( 'Tab Icon Button', 'azo' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_control(
			'selected_icon',
			array(
				'label'            => __( 'Icon', 'azo' ),
				'type'             => Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'default'          => array(
					'value'   => 'fas fa-star',
					'library' => 'fa-solid',
				),
			)
		);

		$this->add_responsive_control(
			'size',
			array(
				'label'     => __( 'Size', 'azo' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min' => 6,
						'max' => 300,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .azo-icon i'   => 'font-size: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .azo-icon svg' => 'width: {{SIZE}}{{UNIT}}; height: auto',
				),
			)
		);

		// Icon Spacing.
		$this->add_responsive_control(
			'button_icon_spacing',
			array(
				'label'           => __( 'Icon Button Space Left', 'azo' ),
				'type'            => Controls_Manager::SLIDER,
				'range'           => array(
					'px' => array(
						'max' => 50,
					),
				),
				'devices'         => array(
					'desktop',
					'tablet',
					'mobile',
				),
				'desktop_default' => array(
					'size' => 29,
					'unit' => 'px',
				),
				'tablet_default'  => array(
					'size' => 0,
					'unit' => 'px',
				),
				'mobile_default'  => array(
					'size' => 0,
					'unit' => 'px',
				),
				'selectors'       => array(
					'{{WRAPPER}} .azo-icon' => 'margin-left: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Render
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$href = '#';
		$attr = '';

		if ( ! empty( $settings['link']['url'] ) ) {
			$href = $settings['link']['url'];

			if ( 'on' === $settings['link']['is_external'] ) {
				$attr .= ' target="_blank"';
			}

			if ( 'on' === $settings['link']['nofollow'] ) {
				$attr .= ' rel="nofollow"';
			}
		}

		$migrated = isset( $settings['__fa4_migrated']['selected_icon'] );
		$is_new   = empty( $settings['icon'] ) && Icons_Manager::is_migration_allowed();
		?>
		<div class="azo-section-heading">
			<h2 class="azo-headings">
				<?php echo esc_html( $settings['widget_title'] ); ?>
			</h2>

			<?php if ( $settings['widget_sub_title'] ) { ?>
				<p class="subtitle">
					<a href="<?php echo esc_url( $href ); ?>" <?php echo wp_kses( $attr, azo_get_allowed_html() ); ?>>
						<?php echo esc_html( $settings['widget_sub_title'] ); ?>

						<span class="azo-icon">
							<?php
							if ( $is_new || $migrated ) {
								Icons_Manager::render_icon( $settings['selected_icon'], array( 'aria-hidden' => 'true' ) );
							} else {
								?>
								<i <?php echo wp_kses( $this->get_render_attribute_string( 'icon' ), azo_get_allowed_html() ); ?>></i>
							<?php } ?>
						</span>
					</a>
				</p>
			<?php } ?>
		</div>
		<?php
	}
}

Plugin::instance()->widgets_manager->register_widget_type( new Azo_Heading() );
