<?php
/**
 * Apar Coundown Widget
 *
 * @package azo
 */

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Apar Coundown
 */
class Azo_Countdown extends Widget_Base {
	/**
	 * Get widget name.
	 *
	 * Retrieve category box widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'azo-countdown';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve category box widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Countdown', 'azo' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve category box widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-countdown';
	}

	/**
	 * Retrieve the list of scripts the widget depended on
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return array( 'azo-countdown' );
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the icon widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return array( 'azo-theme' );
	}

	/**
	 * Register category box widget controls.
	 *
	 * Add different input fields to allow the user to change and customize the widget settings
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function _register_controls() { //phpcs:ignore
		$this->start_controls_section(
			'section_countdown',
			array(
				'label' => esc_html__( 'Countdown', 'azo' ),
			)
		);

		$this->add_control(
			'layout',
			array(
				'label'   => esc_html__( 'Choose Layout', 'azo' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'border'     => esc_html__( 'Border', 'azo' ),
					'normal'     => esc_html__( 'Normal', 'azo' ),
					'no-padding' => esc_html__( 'No Padding', 'azo' ),
				),
				'default' => 'border',
			)
		);

		$this->add_control(
			'message',
			array(
				'label'   => esc_html__( 'Message', 'azo' ),
				'type'    => Controls_Manager::TEXT,
				'default' => esc_html__( 'Ends in:', 'azo' ),
			)
		);

		$this->add_control(
			'due_date',
			array(
				'label'          => esc_html__( 'Due Date', 'azo' ),
				'type'           => Controls_Manager::DATE_TIME,
				'default'        => '10/20/2020',
				'picker_options' => array(
					'dateFormat' => 'm/d/Y',
					'enableTime' => false,
				),
			)
		);

		$this->add_control(
			'label_days',
			array(
				'label'   => esc_html__( 'Days Label', 'azo' ),
				'type'    => Controls_Manager::TEXT,
				'default' => esc_html__( 'Days', 'azo' ),
			)
		);

		$this->add_control(
			'label_hours',
			array(
				'label'   => esc_html__( 'Hours Label', 'azo' ),
				'type'    => Controls_Manager::TEXT,
				'default' => esc_html__( 'Hours', 'azo' ),
			)
		);

		$this->add_control(
			'label_minutes',
			array(
				'label'   => esc_html__( 'Minutes Label', 'azo' ),
				'type'    => Controls_Manager::TEXT,
				'default' => esc_html__( 'Mins', 'azo' ),
			)
		);

		$this->add_control(
			'label_seconds',
			array(
				'label'   => esc_html__( 'Seconds Label', 'azo' ),
				'type'    => Controls_Manager::TEXT,
				'default' => esc_html__( 'Secs', 'azo' ),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_box_style',
			array(
				'label' => esc_html__( 'Boxes', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'alignment',
			array(
				'label'          => esc_html__( 'Alignment', 'azo' ),
				'type'           => Controls_Manager::CHOOSE,
				'options'        => array(
					'flex-start' => array(
						'title' => esc_html__( 'Left', 'azo' ),
						'icon'  => 'fa fa-align-left',
					),
					'center'     => array(
						'title' => esc_html__( 'Center', 'azo' ),
						'icon'  => 'fa fa-align-center',
					),
					'flex-end'   => array(
						'title' => esc_html__( 'Right', 'azo' ),
						'icon'  => 'fa fa-align-right',
					),
				),
				'default'        => 'center',
				'tablet_default' => 'center',
				'mobile_default' => 'center',
				'selectors'      => array(
					'{{WRAPPER}} .countdown-azo-widget' => 'justify-content: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'container_bg_color',
			array(
				'label'     => esc_html__( 'Background Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .countdown-azo-widget .container-box-countdown' => 'background-color: {{VALUE}}',
				),
			)
		);

		$this->add_responsive_control(
			'padding_container',
			array(
				'label'      => esc_html__( 'Padding Container', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .countdown-azo-widget .container-box-countdown' => 'padding:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				),
			)
		);

		$this->add_control(
			'br_container',
			array(
				'label'      => esc_html__( 'Border Radius', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .countdown-azo-widget .container-box-countdown' => 'border-radius:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_content_style',
			array(
				'label' => esc_html__( 'Content', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'message_heading',
			array(
				'label' => esc_html__( 'Message', 'azo' ),
				'type'  => Controls_Manager::HEADING,
			)
		);

		$this->add_control(
			'message_color',
			array(
				'label'     => esc_html__( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .countdown-azo-widget .message' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'message_typography',
				'selector' => '{{WRAPPER}} .countdown-azo-widget .message',
			)
		);

		$this->add_control(
			'heading_digits',
			array(
				'label'     => esc_html__( 'Digits', 'azo' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_control(
			'digits_color',
			array(
				'label'     => esc_html__( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-countdown-digit' => 'color: {{VALUE}}',
					'{{WRAPPER}} .azo-countdown-wrapper.normal .azo-countdown-item:after' => 'color: {{VALUE}}',
				),
				'default'   => '#222222',
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'digits_typography',
				'selector' => '{{WRAPPER}} .azo-countdown-digit',
			)
		);

		$this->add_control(
			'heading_labels',
			array(
				'label'     => esc_html__( 'Labels', 'azo' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_control(
			'labels_color',
			array(
				'label'     => esc_html__( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-countdown-label' => 'color: {{VALUE}}',
				),
				'default'   => '#222222',
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'labels_typography',
				'selector' => '{{WRAPPER}} .azo-countdown-label',
			)
		);

		$this->add_control(
			'no_padding_labels',
			array(
				'label'     => esc_html__( 'No padding', 'azo' ),
				'type'      => Controls_Manager::HEADING,
				'condition' => array(
					'layout' => 'no-padding',
				),
				'separator' => 'before',
			)
		);

		$this->add_control(
			'no_padding_space',
			array(
				'label'     => esc_html__( 'Space', 'azo' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min' => -50,
						'max' => 50,
					),
				),
				'default'   => array(
					'size' => 0,
				),
				'selectors' => array(
					'{{WRAPPER}} .azo-countdown-wrapper.no-padding .azo-countdown-item .azo-countdown-digit' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				),
				'condition' => array(
					'layout' => 'no-padding',
				),
			)
		);

		$this->add_control(
			'border_labels',
			array(
				'label'     => esc_html__( 'Background', 'azo' ),
				'type'      => Controls_Manager::HEADING,
				'condition' => array(
					'layout' => 'border',
				),
				'separator' => 'before',
			)
		);

		$this->add_control(
			'border_color',
			array(
				'label'     => esc_html__( 'Background Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-countdown-wrapper.border .azo-countdown-item' => 'background-color: {{VALUE}}',
				),
				'default'   => '#ffb700',
				'condition' => array(
					'layout' => 'border',
				),
			)
		);

		$this->add_control(
			'dot_labels',
			array(
				'label'     => esc_html__( 'Dot', 'azo' ),
				'type'      => Controls_Manager::HEADING,
				'condition' => array(
					'layout' => 'normal',
				),
				'separator' => 'before',
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'      => 'dot_typography',
				'selector'  => '{{WRAPPER}} .azo-countdown-wrapper.normal .azo-countdown-item:after',
				'condition' => array(
					'layout' => 'normal',
				),
			)
		);

		$this->add_responsive_control(
			'item-space',
			array(
				'label'     => esc_html__( 'Item Spacing', 'azo' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min' => 0,
						'max' => 500,
					),
				),
				'default'   => array(
					'size' => 195,
				),
				'selectors' => array(
					'{{WRAPPER}} .azo-countdown-wrapper.normal .azo-countdown-digit' => 'width:{{SIZE}}{{UNIT}};',
				),
				'condition' => array(
					'layout' => 'normal',
				),
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Render category box widget output on the front end.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
		?>
		<div class="countdown-azo-widget">
			<div class="container-box-countdown">
				<div class="message">
					<?php echo esc_html( $settings['message'] ); ?>
				</div>

				<div class="azo-countdown-wrapper <?php echo esc_attr( $settings['layout'] ); ?>" data-date="<?php echo esc_attr( $settings['due_date'] ); ?>">
					<div class="azo-countdown-item">
						<span id="<?php echo esc_attr( uniqid( 'days-' ) ); ?>" class="azo-countdown-digit azo-count"></span>
						<?php if ( '' !== $settings['label_days'] ) { ?>
							<span class="azo-countdown-label">
								<?php echo esc_html( $settings['label_days'] ); ?>		
							</span>
						<?php } ?>
					</div>

					<div class="azo-countdown-item">
						<span id="<?php echo esc_attr( uniqid( 'hours-' ) ); ?>" class="azo-countdown-digit"></span>
						<?php if ( '' !== $settings['label_hours'] ) { ?>
							<span class="azo-countdown-label">
								<?php echo esc_html( $settings['label_hours'] ); ?>		
							</span>
						<?php } ?>
					</div>

					<div class="azo-countdown-item">
						<span id="<?php echo esc_attr( uniqid( 'minutes-' ) ); ?>" class="azo-countdown-digit"></span>
						<?php if ( '' !== $settings['label_minutes'] ) { ?>
							<span class="azo-countdown-label">
								<?php echo esc_html( $settings['label_minutes'] ); ?>		
							</span>
						<?php } ?>
					</div>

					<div class="azo-countdown-item">
						<span id="<?php echo esc_attr( uniqid( 'seconds-' ) ); ?>" class="azo-countdown-digit"></span>
						<?php if ( '' !== $settings['label_seconds'] ) { ?>
							<span class="azo-countdown-label">
								<?php echo esc_html( $settings['label_seconds'] ); ?>		
							</span>
						<?php } ?>
					</div>
				</div>
			</div>
		</div>
		<?php
	}
}

Plugin::instance()->widgets_manager->register_widget_type( new Azo_Countdown() );
