<?php
/**
 * Azo category slider widget
 *
 * @package azo
 */

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Azo category slider widget.
 *
 * Azo widget that displays an image, a headline, a text and a button.
 *
 * @since 1.0.0
 */
class Azo_Category_Slider extends Widget_Base {
	/**
	 * Get widget name.
	 *
	 * Retrieve category slider widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'azo-category-slider';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve category box widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Azo Category Slider', 'azo' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve category box widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-carousel';
	}

	/**
	 * Retrieve the list of scripts the widget depended on
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return array( 'azo-category-slider' );
	}

	/**
	 * Get categories
	 */
	public function get_categories() {
		return array( 'azo-theme' );
	}

	/**
	 * Register category box widget controls.
	 *
	 * Add different input fields to allow the user to change and customize the widget settings
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() { //phpcs:ignore
		$this->start_controls_section(
			'section_category',
			array(
				'label' => esc_html__( 'Category Slider', 'azo' ),
			)
		);

		$repeater = new Repeater();

		$repeater->add_control(
			'image-slider',
			array(
				'label'   => esc_html__( 'Choose Image', 'azo' ),
				'type'    => Controls_Manager::MEDIA,
				'dynamic' => array(
					'active' => true,
				),
				'default' => array(
					'url' => Utils::get_placeholder_image_src(),
				),
			)
		);

		$repeater->add_control(
			'title-big',
			array(
				'label'       => esc_html__( 'Title Big', 'azo' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter Title Big', 'azo' ),
				'default'     => esc_html__( 'Hifi Audio', 'azo' ),

			)
		);

		$repeater->add_control(
			'description',
			array(
				'label'       => esc_html__( 'Description', 'azo' ),
				'type'        => Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__( 'Enter Description', 'azo' ),
				'default'     => esc_html__( 'Listening to the best', 'azo' ),
			)
		);

		$repeater->add_control(
			'link',
			array(
				'label'       => esc_html__( 'Link to', 'azo' ),
				'type'        => Controls_Manager::URL,
				'dynamic'     => array(
					'active' => true,
				),
				'placeholder' => esc_html__( 'https://your-link.com', 'azo' ),
			)
		);

		$this->add_control(
			'list-category-slider',
			array(
				'label'       => esc_html__( 'List Category', 'azo' ),
				'type'        => Controls_Manager::REPEATER,
				'fields'      => $repeater->get_controls(),
				'title_filed' => '{{{ name }}}',
				'default'     => array(
					array(),
					array(),
					array(),
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_layout',
			array(
				'label' => esc_html__( 'Layout', 'azo' ),
			)
		);

		$this->add_control(
			'text_align',
			array(
				'label'     => esc_html__( 'Alignment', 'azo' ),
				'type'      => Controls_Manager::CHOOSE,
				'options'   => array(
					'left'   => array(
						'title' => esc_html__( 'Left', 'azo' ),
						'icon'  => 'fa fa-align-left',
					),
					'center' => array(
						'title' => esc_html__( 'Center', 'azo' ),
						'icon'  => 'fa fa-align-center',
					),
					'right'  => array(
						'title' => esc_html__( 'Right', 'azo' ),
						'icon'  => 'fa fa-align-right',
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .item-category-info' => 'text-align:{{VALUE}};',
				),
			)
		);

		$this->add_responsive_control(
			'slides_to_show',
			array(
				'label'                 => esc_html__( 'Slides to show', 'azo' ),
				'type'                  => Controls_Manager::NUMBER,
				'min'                   => 1,
				'max'                   => 6,
				'step'                  => 1,
				'devices'               => array( 'desktop', 'tablet', 'mobile' ),
				'slides_to_show_mobile' => array( 'default' => 1 ),
				'slides_to_show_tablet' => array( 'default' => 2 ),
				'slides_to_show'        => array( 'default' => 3 ),
			)
		);

		$this->add_control(
			'dots',
			array(
				'label'        => esc_html__( 'Show Dots', 'azo' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'no',
				'label_on'     => esc_html__( 'No', 'azo' ),
				'label_off'    => esc_html__( 'Yes', 'azo' ),
				'return_value' => 'yes',
			)
		);

		$this->add_control(
			'arrows',
			array(
				'label'        => esc_html__( 'Show Arrows', 'azo' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'yes',
				'label_on'     => esc_html__( 'No', 'azo' ),
				'label_off'    => esc_html__( 'Yes', 'azo' ),
				'return_value' => 'yes',
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_title_big',
			array(
				'label' => esc_html__( 'Title', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'title-big',
				'selector' => '{{WRAPPER}} .title-big',
			)
		);

		$this->add_control(
			'title_big_color',
			array(
				'label'     => esc_html__( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#222222',
				'selectors' => array(
					'{{WRAPPER}} .title-big a' => 'color:{{VALUE}}',
				),
			)
		);

		$this->add_control(
			'title_big_color_hover',
			array(
				'label'     => esc_html__( 'Color Hover', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffb700',
				'selectors' => array(
					'{{WRAPPER}} .title-big a:hover' => 'color:{{VALUE}}',
				),
			)
		);

		$this->add_responsive_control(
			'title_big_padding',
			array(
				'label'      => esc_html__( 'Padding', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .title-big' => 'margin:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_description',
			array(
				'label' => esc_html__( 'Description', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'description',
				'selector' => '{{WRAPPER}} .description',
			)
		);

		$this->add_control(
			'description_color',
			array(
				'label'     => esc_html__( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#333333',
				'selectors' => array(
					'{{WRAPPER}} .description' => 'color:{{VALUE}}',
				),
			)
		);

		$this->add_responsive_control(
			'description_padding',
			array(
				'label'      => esc_html__( 'Padding', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .description' => 'margin:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_arrows',
			array(
				'label' => esc_html__( 'Style Dots', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'dots_color',
			array(
				'label'     => esc_html__( 'Dots Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#222222',
				'selectors' => array(
					'{{WRAPPER}} .category-slider-dots .dots-bullet:after' => 'background-color: {{VALUE}}',
					'{{WRAPPER}} .category-slider-dots .slick-active .dots-bullet:after' => 'border-color: {{VALUE}}',
				),
				'separator' => 'before',
				'condition' => array(
					'dots' => 'yes',
				),
			)
		);

		$this->add_responsive_control(
			'dots_padding',
			array(
				'label'     => esc_html__( 'Margin top', 'azo' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min' => 0,
						'max' => 1000,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .category-slider-dots .slick-dots' => 'margin-top: {{SIZE}}{{UNIT}}',
				),
			)
		);

		$this->end_controls_section();

		// Arrows.
		$this->start_controls_section(
			'product_arrow_style',
			array(
				'label'     => esc_html__( 'Arrows', 'azo' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => array(
					'arrows' => 'yes',
				),
			)
		);

		// TAB START Arrow.
		$this->start_controls_tabs( 'product_arrow_tabs' );

		// Normal.
		$this->start_controls_tab(
			'product_arrow_normal',
			array(
				'label' => __( 'Normal', 'azo' ),
			)
		);

		// Border Color.
		$this->add_control(
			'product_arrow_border_color',
			array(
				'label'     => __( 'Border Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .title-wrappe .arrows-widget .azo-arrow' => 'border-color: {{VALUE}};',
				),
			)
		);

		// Color.
		$this->add_control(
			'product_arrow_color',
			array(
				'label'     => __( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .title-wrappe .arrows-widget .azo-arrow:before' => 'color: {{VALUE}};',
				),
			)
		);

		// Bg Color.
		$this->add_control(
			'product_arrow_bg_hover_color',
			array(
				'label'     => __( 'Background Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .title-wrappe .arrows-widget .azo-arrow' => 'background-color: {{VALUE}};',
				),
			)
		);

		// END NORMAL.
		$this->end_controls_tab();

		// HOVER.
		$this->start_controls_tab(
			'product_arrow_hover',
			array(
				'label' => __( 'Hover', 'azo' ),
			)
		);

		// Border Hover Color.
		$this->add_control(
			'product_arrow_border_hover_color',
			array(
				'label'     => __( 'Border Color Hover', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .title-wrappe .arrows-widget .azo-arrow:hover' => 'border-color: {{VALUE}};',
				),
			)
		);

		// Hover color.
		$this->add_control(
			'product_arrow_hover_color',
			array(
				'label'     => __( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .title-wrappe .arrows-widget .azo-arrow:hover:before' => 'color: {{VALUE}};',
				),
			)
		);

		// Bg Hover Color.
		$this->add_control(
			'product_arrow_bgcolor',
			array(
				'label'     => __( 'Background Hover Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .title-wrappe .arrows-widget .azo-arrow:hover' => 'background-color: {{VALUE}};',
				),
			)
		);

		// TAB END.
		$this->end_controls_tab();
		$this->end_controls_tabs();

		$this->end_controls_section();
	}

	/**
	 * Render
	 */
	public function render() {
		$settings = $this->get_settings_for_display();
		$dot      = $settings['dots'];
		?>
		<div class="azo-slider-category-widget">
			<div class="images-slider-wrapper" data-show="<?php echo esc_attr( $settings['slides_to_show'] ); ?>" data-show-tablet="<?php echo esc_attr( $settings['slides_to_show_tablet'] ); ?>" data-show-mobile="<?php echo esc_attr( $settings['slides_to_show_mobile'] ); ?>" >

				<?php if ( 'yes' === $settings['arrows'] ) { ?>
					<div class="title-wrappe arrowpreset2">
						<div class="arrows-widget">
							<button class="azo-arrow arrows-prev"></button>
							<button class="azo-arrow arrows-next"></button>
						</div>
					</div>
				<?php } ?>

				<div class="js-category-slideshow list-category" data-dot="<?php echo esc_attr( $dot ); ?>">
					<?php foreach ( $settings['list-category-slider'] as $category_item ) { ?>
						<div class="azo-slider-item-category">
							<div class="box-images-slider">
								<a href="<?php echo esc_url( $category_item['link']['url'] ); ?>">
									<img src="<?php echo esc_url( $category_item['image-slider']['url'] ); ?>" alt="<?php esc_attr_e( 'image-category', 'azo' ); ?>" class="slider-images">
								</a>
							</div>
							<div class="item-category-info">
								<?php if ( ! empty( $category_item['title-big'] ) ) { ?>									
									<h4 class="title-big">
										<a href="<?php echo esc_url( $category_item['link']['url'] ); ?>">
											<?php echo esc_html( $category_item['title-big'] ); ?>
										</a>		
									</h4>									
								<?php } ?>
								<?php if ( ! empty( $category_item['description'] ) ) { ?>
									<p class="description"><?php echo esc_html( $category_item['description'] ); ?></p>
								<?php } ?>
							</div>
						</div>
					<?php } ?>
				</div>
				<div class="category-slider-dots"></div>
			</div>
		</div>
		<?php
	}
}

Plugin::instance()->widgets_manager->register_widget_type( new Azo_Category_Slider() );
