<?php
/**
 * Azo category box widget
 *
 * @package azo
 */

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Azo category box widget.
 *
 * Azo widget that displays an image, a headline, a text and a button.
 *
 * @since 1.0.0
 */
class Azo_Category_Box extends Widget_Base {
	/**
	 * Get widget name.
	 *
	 * Retrieve category box widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'azo-category-box';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve category box widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Category Box', 'azo' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve category box widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-info-box';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the icon widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return array( 'azo-theme' );
	}

	/**
	 * Register category box widget controls.
	 *
	 * Add different input fields to allow the user to change and customize the widget settings
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() { //phpcs:ignore
		$this->start_controls_section(
			'section_category',
			array(
				'label' => esc_html__( 'Category Box', 'azo' ),
			)
		);

		$this->add_control(
			'hover_img',
			array(
				'label'   => esc_html__( 'Hover Image', 'azo' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'effect_1' => esc_html__( 'Effect 1', 'azo' ),
					'effect_2' => esc_html__( 'Effect 2', 'azo' ),
				),
				'default' => 'effect_1',
			)
		);

		$this->add_control(
			'hover_description',
			array(
				'label'   => esc_html__( 'Hover Description', 'azo' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'ef_description' => esc_html__( 'Effect', 'azo' ),
					'ef_none'        => esc_html__( 'None', 'azo' ),
				),
				'default' => 'ef_description',
			)
		);

		$this->add_control(
			'image',
			array(
				'label'   => esc_html__( 'Choose Image', 'azo' ),
				'type'    => Controls_Manager::MEDIA,
				'dynamic' => array(
					'active' => true,
				),
				'default' => array(
					'url' => Utils::get_placeholder_image_src(),
				),
			)
		);

		$this->add_group_control(
			Group_Control_Image_Size::get_type(),
			array(
				'name'      => 'thumbnail',
				'default'   => 'full',
				'separator' => 'none',
			)
		);

		$this->add_control(
			'title_text',
			array(
				'label'       => esc_html__( 'Title', 'azo' ),
				'type'        => Controls_Manager::TEXT,
				'dynamic'     => array(
					'active' => true,
				),
				'default'     => esc_html__( 'This is the heading', 'azo' ),
				'placeholder' => esc_html__( 'Enter your text title', 'azo' ),
				'label_block' => true,
			)
		);

		$this->add_control(
			'title_size',
			array(
				'label'   => esc_html__( 'Title HTML tag', 'azo' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'h1'   => 'H1',
					'h2'   => 'H2',
					'h3'   => 'H3',
					'h4'   => 'H4',
					'h5'   => 'H5',
					'div'  => 'div',
					'span' => 'span',
					'p'    => 'p',
				),
				'default' => 'h3',
			)
		);

		$this->add_control(
			'description',
			array(
				'label'   => esc_html__( 'Description', 'azo' ),
				'type'    => Controls_Manager::TEXT,
				'default' => esc_html__( '102 Iteam', 'azo' ),
			)
		);

		$this->add_control(
			'link',
			array(
				'label'       => esc_html__( 'Link to', 'azo' ),
				'type'        => Controls_Manager::URL,
				'dynamic'     => array(
					'active' => true,
				),
				'placeholder' => esc_html__( 'https://your-link.com', 'azo' ),
				'separator'   => 'before',
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_content',
			array(
				'label' => esc_html__( 'Content', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'border_color_image',
			array(
				'label'     => esc_html__( 'Color border hover', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#fff',
				'selectors' => array(
					'{{WRAPPER}} .azo-category-box-image.hover-effect_2:before' => 'border-color: {{VALUE}}',
					'{{WRAPPER}} .azo-category-box-image.hover-effect_2:after' => 'border-color: {{VALUE}}',
				),
				'condition' => array(
					'hover_img' => 'effect_2',
				),
			)
		);

		$this->add_control(
			'text_align',
			array(
				'label'     => esc_html__( 'Alignment', 'azo' ),
				'type'      => Controls_Manager::CHOOSE,
				'options'   => array(
					'left'   => array(
						'title' => esc_html__( 'Left', 'azo' ),
						'icon'  => 'fa fa-align-left',
					),
					'center' => array(
						'title' => esc_html__( 'Center', 'azo' ),
						'icon'  => 'fa fa-align-center',
					),
					'right'  => array(
						'title' => esc_html__( 'Right', 'azo' ),
						'icon'  => 'fa fa-align-right',
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .azo-category-box-image' => 'text-align:{{VALUE}};',
					'{{WRAPPER}} .azo-category-box-content' => 'text-align:{{VALUE}};',
				),
			)
		);

		$this->add_control(
			'content_title',
			array(
				'label'     => esc_html__( 'Title', 'azo' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'title_typography',
				'selector' => '{{WRAPPER}} .azo-category-box-title a',
			)
		);

		$this->add_control(
			'title_color',
			array(
				'label'     => esc_html__( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#2f2f30',
				'selectors' => array(
					'{{WRAPPER}} .azo-category-box-title a' => 'color:{{VALUE}};',
				),
			)
		);

		$this->add_control(
			'title_color_hover',
			array(
				'label'     => esc_html__( 'Color Hover', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-category-box-wrapper:hover .azo-category-box-title a' => 'color:{{VALUE}};',
				),
			)
		);

		$this->add_responsive_control(
			'title_margin',
			array(
				'label'      => esc_html__( 'Margin', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .azo-category-box-title' => 'margin:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				),
			)
		);

		$this->add_control(
			'description_title',
			array(
				'label'     => esc_html__( 'Description', 'azo' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_group_control(
			Group_Control_typography::get_type(),
			array(
				'name'     => 'description_typography',
				'selector' => '{{WRAPPER}} .azo-category-description',
			)
		);

		$this->add_control(
			'description_color',
			array(
				'label'     => esc_html__( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-category-description' => 'color:{{VALUE}};',
				),
				'default'   => '#959595',
			)
		);

		$this->add_responsive_control(
			'description_margin',
			array(
				'label'      => esc_html__( 'Margin', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .azo-category-description' => 'margin:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				),
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Render category box widget output on the front end.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$image      = $settings['image'];
		$descrip    = $settings['description'];
		$image_url  = $image['url'];
		$image_alt  = azo_img_alt( $image['id'], esc_attr__( 'Category Box Image', 'azo' ) );
		$title_text = $settings['title_text'];

		$href = '#';
		$attr = '';

		if ( ! empty( $settings['link']['url'] ) ) {
			$href = $settings['link']['url'];

			if ( 'on' === $settings['link']['is_external'] ) {
				$attr .= ' target="_blank"';
			}

			if ( 'on' === $settings['link']['nofollow'] ) {
				$attr .= ' rel="nofollow"';
			}
		}
		?>
		<div class="azo-category-box-wrapper">
			<?php if ( 'effect_1' === $settings['hover_img'] ) { ?>
				<div class="azo-category-box-image hover-<?php echo esc_attr( $settings['hover_img'] ); ?>">
					<a href="<?php echo esc_url( $href ); ?>" <?php echo wp_kses( $attr, azo_get_allowed_html() ); ?> class="category-link">
						<img src="<?php echo esc_url( $image_url ); ?>" alt="<?php echo esc_attr( $image_alt ); ?>">
					</a>
				</div>
			<?php } elseif ( 'effect_2' === $settings['hover_img'] ) { ?>
				<div class="azo-category-box-image hover-<?php echo esc_attr( $settings['hover_img'] ); ?>">
					<a href="<?php echo esc_url( $href ); ?>" <?php echo wp_kses( $attr, azo_get_allowed_html() ); ?> class="azo-category-box-link"></a>
						<img src="<?php echo esc_url( $image_url ); ?>" alt="<?php echo esc_attr( $image_alt ); ?>">
				</div>
			<?php } ?>

			<div class="azo-category-box-content">
				<?php if ( ! empty( $title_text ) ) { ?>
				<h3 class="azo-category-box-title">
					<a href="<?php echo esc_url( $href ); ?>" <?php echo wp_kses( $attr, azo_get_allowed_html() ); ?>>
						<?php echo wp_kses( $title_text, azo_get_allowed_html() ); ?>
					</a>
				</h3>
				<?php } ?>

				<?php if ( ! empty( $descrip ) ) { ?>
				<p class="azo-category-description effect-<?php echo esc_attr( $settings['hover_description'] ); ?>">
					<?php echo wp_kses( $descrip, azo_get_allowed_html() ); ?>
				</p>
				<?php } ?>
			</div>
		</div>
		<?php
	}
}

Plugin::instance()->widgets_manager->register_widget_type( new Azo_Category_Box() );
