<?php
/**
 * Azo Cart Widget
 *
 * @package azo
 */

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! defined( 'WC_PLUGIN_FILE' ) ) {
	return;
}

/**
 * Class for azo elementor product images widget.
 */
class Azo_Cart extends \Elementor\Widget_Base {
	/**
	 * Category
	 */
	public function get_categories() {
		return array( 'azo-theme' );
	}

	/**
	 * Name
	 */
	public function get_name() {
		return 'azo-cart';
	}

	/**
	 * Gets the title.
	 */
	public function get_title() {
		return __( 'Cart Azo', 'azo' );
	}

	/**
	 * Gets the icon.
	 */
	public function get_icon() {
		return 'eicon-cart';
	}

	/**
	 * Controls
	 */
	protected function register_controls() { //phpcs:ignore
		$this->content();
		$this->cart_icon_style();
	}

	/**
	 * Content
	 */
	protected function content() {
		$this->start_controls_section(
			'cart_icon',
			array(
				'label' => __( 'Cart Icon', 'azo' ),
			)
		);

		$this->add_control(
			'show_subtotal',
			array(
				'label'        => __( 'Subtotal', 'azo' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => __( 'Show', 'azo' ),
				'label_off'    => __( 'Hide', 'azo' ),
				'return_value' => 'yes',
				'default'      => 'yes',
			)
		);

		$this->add_control(
			'content_align',
			array(
				'label'     => esc_html__( 'Alignment', 'azo' ),
				'type'      => Controls_Manager::CHOOSE,
				'options'   => array(
					'left'   => array(
						'title' => esc_html__( 'Left', 'azo' ),
						'icon'  => 'fa fa-align-left',
					),
					'center' => array(
						'title' => esc_html__( 'Center', 'azo' ),
						'icon'  => 'fa fa-align-center',
					),
					'right'  => array(
						'title' => esc_html__( 'Right', 'azo' ),
						'icon'  => 'fa fa-align-right',
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .cart-azo-elementor' => 'text-align:{{VALUE}}',
				),
				'default'   => 'left',
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Style
	 */
	protected function cart_icon_style() {
		$this->start_controls_section(
			'style_icon',
			array(
				'label' => __( 'Cart Icon', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'subtotal_heading',
			array(
				'label'     => __( 'Subtotal', 'azo' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => array(
					'show_subtotal' => 'yes',
				),
			)
		);

		$this->add_control(
			'color_subtotal',
			array(
				'label'     => __( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#2f2f30',
				'selectors' => array(
					'{{WRAPPER}} .azo-subtotal .woocommerce-Price-amount' => 'color: {{VALUE}}',
				),
				'condition' => array(
					'show_subtotal' => 'yes',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'      => 'subtotal_typography',
				'label'     => __( 'Typography', 'azo' ),
				'selector'  => '{{WRAPPER}} .azo-subtotal .woocommerce-Price-amount',
				'condition' => array(
					'show_subtotal' => 'yes',
				),
			)
		);

		$this->add_responsive_control(
			'space',
			array(
				'label'      => __( 'Space', 'azo' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'min'  => 00,
						'max'  => 50,
						'step' => 1,
					),
				),
				'selectors'  => array(
					'{{WRAPPER}} .azo-subtotal .woocommerce-Price-amount' => 'margin-left: {{SIZE}}{{UNIT}};',
				),
				'condition'  => array(
					'show_subtotal' => 'yes',
				),
			)
		);

		$this->add_control(
			'icon_heading',
			array(
				'label'     => __( 'Icon', 'azo' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_control(
			'color_icon',
			array(
				'label'     => __( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#2f2f30',
				'selectors' => array(
					'{{WRAPPER}} .cart-custom-azo .icoros-icon_shopping_cart.js-cart-button:before' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'color_icon_hover',
			array(
				'label'     => __( 'Color Hover', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffb700',
				'selectors' => array(
					'{{WRAPPER}} .cart-custom-azo:hover .icoros-icon_shopping_cart.js-cart-button:before' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_responsive_control(
			'font_size',
			array(
				'label'      => __( 'Font Size', 'azo' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'min'  => 10,
						'max'  => 50,
						'step' => 1,
					),
				),
				'selectors'  => array(
					'{{WRAPPER}} .cart-custom-azo .js-cart-button' => 'font-size: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->add_control(
			'icon_counter_heading',
			array(
				'label'     => __( 'Counter', 'azo' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_control(
			'color_icon_counter',
			array(
				'label'     => __( 'Color Counter', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#222222',
				'selectors' => array(
					'{{WRAPPER}} .cart-custom-azo .shop-cart-count' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'icon_counter_typography',
				'label'    => __( 'Typography', 'azo' ),
				'selector' => '{{WRAPPER}} .cart-custom-azo .shop-cart-count',
			)
		);

		$this->add_control(
			'icon_counter_background',
			array(
				'label'     => __( 'Background', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffb700',
				'selectors' => array(
					'{{WRAPPER}} .cart-custom-azo .shop-cart-count' => 'background-color: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'distance-top',
			array(
				'label'      => __( 'Distance Top', 'azo' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'min'  => -50,
						'max'  => 50,
						'step' => 1,
					),
				),
				'selectors'  => array(
					'{{WRAPPER}} .cart-custom-azo .shop-cart-count' => 'top: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->add_control(
			'distance-right',
			array(
				'label'      => __( 'Distance Right', 'azo' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'min'  => -50,
						'max'  => 50,
						'step' => 1,
					),
				),
				'selectors'  => array(
					'{{WRAPPER}} .cart-custom-azo .shop-cart-count' => 'right: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Render
	 */
	public function render() {
		$settings = $this->get_settings_for_display();

		global $woocommerce;

		$product_count = $woocommerce->cart->get_cart_contents_count();
		$sub_total     = $woocommerce->cart->get_cart_subtotal();
		?>
		<div class="cart-azo-elementor">
			<?php if ( 'yes' === $settings['show_subtotal'] ) { ?>
				<span class="azo-subtotal">
					<?php echo wp_kses( $sub_total, azo_get_allowed_html() ); ?>
				</span>
			<?php } ?>

			<span class="cart-custom-azo">
				<a href="<?php echo esc_url( wc_get_cart_url() ); ?>" class="js-cart-button">
					<?php azo_get_svg( 'cart' ); ?>
				</a>
				<span class="shop-cart-count"><?php echo esc_html( $product_count ); ?></span>
			</span>
		</div>
		<?php
	}
}

Plugin::instance()->widgets_manager->register_widget_type( new Azo_Cart() );
