<?php
/**
 * Azo Blog Slider
 *
 * @package azo
 */

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Azo blog slider widget.
 *
 * Azo widget that displays blog slider.
 *
 * @since 1.0.0
 */
class Azo_Blog_Slider extends Widget_Base {
	/**
	 * Get widget name.
	 *
	 * Retrieve blog slider widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'azo-blog-slider';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve blog posts widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Blog Slider', 'azo' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve blog posts widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-post-slider';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the icon widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return array( 'azo-theme' );
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return array( 'azo-blog-slider' );
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 */
	protected function register_controls() { //phpcs:ignore
		$this->start_controls_section(
			'section_blog_slider',
			array(
				'label' => esc_html__( 'Blog Slider', 'azo' ),
			)
		);

		$this->add_control(
			'title',
			array(
				'label'   => __( 'Title', 'azo' ),
				'type'    => Controls_Manager::TEXT,
				'default' => 'Most Popular Blog Posts',
			)
		);

		// Alignment.
		$this->add_responsive_control(
			'align',
			array(
				'label'   => esc_html__( 'Alignment', 'azo' ),
				'type'    => Controls_Manager::CHOOSE,
				'options' => array(
					'left'   => array(
						'title' => esc_html__( 'Left', 'azo' ),
						'icon'  => 'fa fa-align-left',
					),
					'center' => array(
						'title' => esc_html__( 'Center', 'azo' ),
						'icon'  => 'fa fa-align-center',
					),
					'right'  => array(
						'title' => esc_html__( 'Right', 'azo' ),
						'icon'  => 'fa fa-align-right',
					),
				),
				'default' => 'center',
			)
		);

		$this->add_control(
			'limit',
			array(
				'label'   => esc_html__( 'Posts Per Page', 'azo' ),
				'type'    => Controls_Manager::NUMBER,
				'default' => 4,
			)
		);

		// Slides to show.
		$this->add_responsive_control(
			'slides_to_show',
			array(
				'label'                 => esc_html__( 'Slides to show', 'azo' ),
				'type'                  => Controls_Manager::NUMBER,
				'min'                   => 1,
				'max'                   => 5,
				'step'                  => 1,
				'devices'               => array( 'desktop', 'tablet', 'mobile' ),
				'slides_to_show_mobile' => array(
					'default' => 1,
				),
				'slides_to_show_tablet' => array(
					'default' => 2,
				),
				'slides_to_show'        => array(
					'default' => 2,
				),
			)
		);

		$this->add_control(
			'limit_text',
			array(
				'label'   => esc_html__( 'Limit Text', 'azo' ),
				'type'    => Controls_Manager::NUMBER,
				'default' => 15,
			)
		);

		$this->add_control(
			'date',
			array(
				'label'        => esc_html__( 'Show Date', 'azo' ),
				'type'         => Controls_Manager::SWITCHER,
				'separator'    => 'before',
				'default'      => 'yes',
				'on'           => esc_html__( 'No', 'azo' ),
				'off'          => esc_html__( 'Yes', 'azo' ),
				'return_value' => 'yes',
			)
		);

		$this->add_control(
			'category',
			array(
				'label'        => esc_html__( 'Show Category', 'azo' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'yes',
				'on'           => esc_html__( 'No', 'azo' ),
				'off'          => esc_html__( 'Yes', 'azo' ),
				'return_value' => 'yes',
			)
		);

		$this->add_control(
			'info',
			array(
				'label'        => esc_html__( 'Show Comment', 'azo' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'yes',
				'on'           => esc_html__( 'No', 'azo' ),
				'off'          => esc_html__( 'Yes', 'azo' ),
				'return_value' => 'yes',
			)
		);

		$this->add_control(
			'excerpt',
			array(
				'label'        => esc_html__( 'Show Excerpt', 'azo' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'yes',
				'on'           => esc_html__( 'No', 'azo' ),
				'off'          => esc_html__( 'Yes', 'azo' ),
				'return_value' => 'yes',
			)
		);

		$this->add_control(
			'readmore',
			array(
				'label'        => esc_html__( 'Show Read More', 'azo' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'yes',
				'on'           => esc_html__( 'No', 'azo' ),
				'off'          => esc_html__( 'Yes', 'azo' ),
				'return_value' => 'yes',
			)
		);

		$this->add_control(
			'dots',
			array(
				'label'        => esc_html__( 'Show Dots', 'azo' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'no',
				'label_on'     => esc_html__( 'No', 'azo' ),
				'label_off'    => esc_html__( 'Yes', 'azo' ),
				'return_value' => 'yes',
			)
		);

		$this->add_control(
			'arrow',
			array(
				'label'        => esc_html__( 'Show Arrow', 'azo' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'yes',
				'label_on'     => esc_html__( 'No', 'azo' ),
				'label_off'    => esc_html__( 'Yes', 'azo' ),
				'return_value' => 'yes',
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'blog',
			array(
				'label' => __( 'Heading Blog', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'heading_color',
			array(
				'label'     => esc_html__( 'Color Heading', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#222222',
				'selectors' => array(
					'{{WRAPPER}} .title-blog-slider' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'heading_typo',
				'selector' => '{{WRAPPER}} .title-blog-slider',
			)
		);

		$this->end_controls_section();

		// Arrows.
		$this->start_controls_section(
			'product_arrow_style',
			array(
				'label'     => esc_html__( 'Arrows', 'azo' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => array(
					'arrow' => 'yes',
				),
			)
		);

		// TAB START Arrow.
		$this->start_controls_tabs( 'product_arrow_tabs' );

		// Normal.
		$this->start_controls_tab(
			'product_arrow_normal',
			array(
				'label' => __( 'Normal', 'azo' ),
			)
		);

		// Border Color.
		$this->add_control(
			'product_arrow_border_color',
			array(
				'label'     => __( 'Border Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .title-wrappe .arrows-widget .azo-arrow' => 'border-color: {{VALUE}};',
				),
			)
		);

		// Color.
		$this->add_control(
			'product_arrow_color',
			array(
				'label'     => __( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .title-wrappe .arrows-widget .azo-arrow:before' => 'color: {{VALUE}};',
				),
			)
		);

		// Bg Color.
		$this->add_control(
			'product_arrow_bg_hover_color',
			array(
				'label'     => __( 'Background Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .title-wrappe .arrows-widget .azo-arrow' => 'background-color: {{VALUE}};',
				),
			)
		);

		// END NORMAL.
		$this->end_controls_tab();

		// HOVER.
		$this->start_controls_tab(
			'product_arrow_hover',
			array(
				'label' => __( 'Hover', 'azo' ),
			)
		);

		// Border Hover Color.
		$this->add_control(
			'product_arrow_border_hover_color',
			array(
				'label'     => __( 'Border Color Hover', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .title-wrappe .arrows-widget .azo-arrow:hover' => 'border-color: {{VALUE}};',
				),
			)
		);

		// Hover color.
		$this->add_control(
			'product_arrow_hover_color',
			array(
				'label'     => __( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .title-wrappe .arrows-widget .azo-arrow:hover:before' => 'color: {{VALUE}};',
				),
			)
		);

		// Bg Hover Color.
		$this->add_control(
			'product_arrow_bgcolor',
			array(
				'label'     => __( 'Background Hover Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .title-wrappe .arrows-widget .azo-arrow:hover' => 'background-color: {{VALUE}};',
				),
			)
		);

		// TAB END.
		$this->end_controls_tab();
		$this->end_controls_tabs();

		$this->end_controls_section();

		$this->start_controls_section(
			'style_item_blog',
			array(
				'label' => esc_html__( 'Item', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_responsive_control(
			'height_img',
			array(
				'label'     => esc_html__( 'Height Image', 'azo' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min' => 0,
						'max' => 1000,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .blog-entry-thumbnail img' => 'height: {{SIZE}}{{UNIT}}',
				),
			)
		);

		$this->add_control(
			'br-image',
			array(
				'label'      => esc_html__( 'Border Radius Images', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .blog-entry-thumbnail img' => 'border-radius:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				),
			)
		);

		$this->add_control(
			'br-info',
			array(
				'label'      => esc_html__( 'Border Radius Info', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .blog-box-slider' => 'border-radius:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				),
			)
		);

		$this->add_responsive_control(
			'item_padding',
			array(
				'label'      => esc_html__( 'Padding info', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .azo-blog-slider-widget .blog-detail-box-slider' => 'padding:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				),
			)
		);

		$this->start_controls_tabs(
			'style_info_hover'
		);

		$this->start_controls_tab(
			'tab_title_normal',
			array(
				'label' => esc_html__( 'Nomal', 'azo' ),
			)
		);

		$this->add_control(
			'bgcolor_info',
			array(
				'label'     => esc_html__( 'Background', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .blog-box-slider' => 'background-color: {{VALUE}}',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'label'    => esc_html__( 'Box Shadow', 'azo' ),
				'name'     => 'box_shadow_blog_slider',
				'selector' => '{{WRAPPER}} .blog-box-slider',
			)
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tab_title_hover',
			array(
				'label' => esc_html__( 'Hover', 'azo' ),
			)
		);

		$this->add_control(
			'bgcolor_info_hover',
			array(
				'label'     => esc_html__( 'Background', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .blog-box-slider:hover' => 'background-color: {{VALUE}}',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'label'    => esc_html__( 'Box Shadow', 'azo' ),
				'name'     => 'box_shadow_blog_slider_hover',
				'selector' => '{{WRAPPER}} .blog-box-slider:hover',
			)
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();

		$this->start_controls_section(
			'section_date',
			array(
				'label'     => esc_html__( 'Date', 'azo' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => array(
					'date' => 'yes',
				),
			)
		);

		$this->add_control(
			'date_color',
			array(
				'label'     => esc_html__( 'Date Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#222222',
				'selectors' => array(
					'{{WRAPPER}} .azo-blog-slider-widget .blog-entry-thumbnail .azo-date-time'  => 'color: {{VALUE}}',
					'{{WRAPPER}} .blog-entry-thumbnail .azo-date-time span.space'  => 'background-color: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'date_bg_color',
			array(
				'label'     => esc_html__( 'Date Background Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffb700',
				'selectors' => array(
					'{{WRAPPER}} .azo-blog-slider-widget .blog-entry-thumbnail .azo-date-time'  => 'background-color: {{VALUE}}',
				),
			)
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			array(
				'label'    => esc_html__( 'Typography Date', 'azo' ),
				'name'     => 'date_typography',
				'selector' => '{{WRAPPER}} .azo-blog-slider-widget .blog-entry-thumbnail .azo-date-time .day',
			)
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			array(
				'label'    => esc_html__( 'Typography Month', 'azo' ),
				'name'     => 'month_typography',
				'selector' => '{{WRAPPER}} .azo-blog-slider-widget .blog-entry-thumbnail .azo-date-time .month',
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_title_style',
			array(
				'label' => esc_html__( 'Title', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'title_color',
			array(
				'label'     => esc_html__( 'Color Title', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .blog-entry-title a' => 'color: {{VALUE}}',
				),
				'default'   => '#222222',
			)
		);

		$this->add_control(
			'title_hover_color',
			array(
				'label'     => esc_html__( 'Color Hover Title', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffb700',
				'selectors' => array(
					'{{WRAPPER}} .blog-entry-title a:hover' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			array(
				'name'     => 'title_typography',
				'selector' => '{{WRAPPER}} h3.blog-entry-title',
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_cate_comment_style',
			array(
				'label' => esc_html__( 'Category, commment', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'info_color',
			array(
				'label'     => esc_html__( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#e76700',
				'selectors' => array(
					'{{WRAPPER}} .blog-header-info .if-item a'  => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'info_hover_color',
			array(
				'label'     => esc_html__( 'Hover Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffb700',
				'selectors' => array(
					'{{WRAPPER}} .blog-header-info .if-item a:hover'  => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			array(
				'label'    => esc_html__( 'Typography Info', 'azo' ),
				'name'     => 'category_typography',
				'selector' => '{{WRAPPER}} .blog-header-info .if-item a',
			)
		);

		$this->add_responsive_control(
			'margin-top-comment',
			array(
				'label'     => esc_html__( 'Margin Top', 'azo' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min' => 0,
						'max' => 1000,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .blog-header-info' => 'margin-top: {{SIZE}}{{UNIT}}',
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'description_style',
			array(
				'label'     => esc_html__( 'Except', 'azo' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => array(
					'excerpt' => 'yes',
				),
			)
		);

		$this->add_control(
			'except_color',
			array(
				'label'     => esc_html__( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#767676',
				'selectors' => array(
					'{{WRAPPER}} .blog-post-excerpt' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			array(
				'name'     => 'except_typography',
				'selector' => '{{WRAPPER}} .blog-post-excerpt',
			)
		);

		$this->add_responsive_control(
			'margin-top-excerpt',
			array(
				'label'     => esc_html__( 'Margin Top', 'azo' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min' => 0,
						'max' => 1000,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .blog-post-excerpt' => 'margin-top: {{SIZE}}{{UNIT}}',
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'button_style',
			array(
				'label'     => esc_html__( 'Button', 'azo' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => array(
					'readmore' => 'yes',
				),
			)
		);

		$this->add_control(
			'read_more_color',
			array(
				'label'     => esc_html__( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .blog-read-more a' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'read_more_color_hover',
			array(
				'label'     => esc_html__( 'Color Hover', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .blog-read-more a:hover' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			array(
				'name'     => 'read_more_typography',
				'selector' => '{{WRAPPER}} .blog-read-more a',
			)
		);

		$this->add_responsive_control(
			'margin-top-button',
			array(
				'label'     => esc_html__( 'Margin Top', 'azo' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min' => 0,
						'max' => 1000,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .blog-read-more' => 'margin-top: {{SIZE}}{{UNIT}}',
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'dost_style',
			array(
				'label'     => esc_html__( 'Dots', 'azo' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => array(
					'dots'  => 'yes',
					'arrow' => 'yes',
				),

			)
		);

		$this->add_control(
			'dots_color',
			array(
				'label'     => esc_html__( 'Dots Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#222222',
				'selectors' => array(
					'{{WRAPPER}} .blog-slider-dots .dots-bullet:after' => 'background-color: {{VALUE}}',
					'{{WRAPPER}} .blog-slider-dots .slick-active .dots-bullet:after' => 'border-color: {{VALUE}}',
				),
				'separator' => 'before',
				'condition' => array(
					'dots' => 'yes',
				),
			)
		);

		$this->add_responsive_control(
			'margin-top-dots',
			array(
				'label'     => esc_html__( 'Margin Top', 'azo' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min' => 0,
						'max' => 1000,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .blog-slider-dots' => 'margin-top: {{SIZE}}{{UNIT}}',
				),
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$paged   = ( get_query_var( 'paged' ) ) ? get_query_var( 'paged' ) : 1;
		$dot     = $settings['dots'];
		$arrow   = $settings['arrow'];
		$limited = $settings['limit_text'];
		$args    = array(
			'post_type'           => 'post',
			'posts_per_page'      => $settings['limit'],
			'paged'               => $paged,
			'ignore_sticky_posts' => 1,
		);

		$posts = new \WP_Query( $args );

		?>

		<div class="azo-blog-slider-widget">
			<div class="blog-slider-wrapper" data-show="<?php echo esc_attr( $settings['slides_to_show'] ); ?>" data-show-tablet="<?php echo esc_attr( $settings['slides_to_show_tablet'] ); ?>" data-show-mobile="<?php echo esc_attr( $settings['slides_to_show_mobile'] ); ?>" >
				<div class="title-wrappe">
					<h2 class="title-blog-slider">
						<?php echo esc_html( $settings['title'] ); ?>
					</h2>

					<?php if ( 'yes' === $arrow ) { ?>
						<div class="arrows-widget">
							<button class="azo-arrow arrows-prev"></button>
							<button class="azo-arrow arrows-next"></button>
						</div>
					<?php } ?>
				</div>

				<div class="js-blog-slider blog-slider-list" data-dot="<?php echo esc_attr( $dot ); ?>">
					<?php
					while ( $posts->have_posts() ) {
						$posts->the_post();
						?>
						<article id="post-<?php the_ID(); ?>" class="post-blog-slider">
							<div class="blog-box-slider <?php echo esc_attr( $settings['excerpt'] ); ?>no-excerpt">
								<div class="blog-entry-thumbnail">
									<a href="<?php echo esc_url( the_permalink() ); ?>">
										<?php
										if ( has_post_thumbnail() ) {
											the_post_thumbnail();
										}
										?>
										<?php
										if ( 'yes' === $settings['date'] ) {
											?>
											<div class="azo-date-time">
												<?php echo esc_html( azo_date_format() ); ?>
											</div>
										<?php } ?>
									</a>
								</div>

								<div class="blog-entry-header entry-wrapper blog-detail-box-slider <?php echo esc_attr( $settings['align'] ); ?>">
									<div class="blog-header-info">
										<?php if ( 'yes' === $settings['category'] ) { ?>
											<span class="if-item if-cat">
												<?php echo wp_kses( azo_blog_categories(), azo_get_allowed_html() ); ?>
											</span>
										<?php } ?>
										<?php if ( 'yes' === $settings['info'] ) { ?>
											<span class="if-item if-commnet">
												<?php azo_comment_count(); ?>
											</span>
										<?php } ?>
									</div>

									<h3 class="blog-entry-title">
										<a href="<?php echo esc_url( the_permalink() ); ?>">
											<?php echo esc_html( get_the_title() ); ?>
										</a>
									</h3>

									<?php if ( 'yes' === $settings['excerpt'] ) { ?>
										<div class="blog-post-excerpt">
											<?php echo esc_html( wp_trim_words( get_the_content( get_the_ID() ), $limited, null ) ); ?>
										</div>
									<?php } ?>

									<?php if ( 'yes' === $settings['readmore'] ) { ?>
										<div class="blog-read-more">
											<a href="<?php echo esc_url( the_permalink() ); ?>" class="read-more">
												<?php echo esc_html__( 'Continue Reading', 'azo' ); ?>
											</a>
										</div>
									<?php } ?>
								</div>
							</div>
						</article>
					<?php } ?>
				</div>
				<div class="blog-slider-dots"></div>
			</div>
		</div>

		<?php
		wp_reset_postdata();
	}
}

Plugin::instance()->widgets_manager->register_widget_type( new Azo_Blog_Slider() );
