<?php
/**
 * Azo Blog Post
 *
 * @package azo
 */

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Azo blog posts widget.
 *
 * Azo widget that displays blog posts.
 *
 * @since 1.0.0
 */
class Azo_Blog_Posts extends Widget_Base {
	/**
	 * Get widget name.
	 *
	 * Retrieve blog posts widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'azo-blog-posts';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve blog posts widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Blog Posts', 'azo' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve blog posts widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-post-list';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the icon widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return array( 'azo-theme' );
	}

	/**
	 * Register blog posts widget controls.
	 *
	 * Add different input fields to allow the user to change and customize the widget settings
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() { //phpcs:ignore
		parent::register_controls();

		$this->register_layout_section_controls();
		$this->register_query_section_controls();
		$this->register_pagination_section_controls();
	}

	/**
	 * Register layout section controls
	 */
	protected function register_layout_section_controls() {
		$this->start_controls_section(
			'section_layout',
			array(
				'label' => esc_html__( 'Layout', 'azo' ),
			)
		);

		$this->add_control(
			'blog_preset',
			array(
				'label'   => esc_html__( 'Blog Preset', 'azo' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'blogpreset1' => esc_html__( 'Blog Preset 1', 'azo' ),
					'blogpreset2' => esc_html__( 'Blog Preset 2', 'azo' ),
					'blogpreset3' => esc_html__( 'Blog Preset 3', 'azo' ),
				),
				'default' => 'blogpreset1',
			)
		);

		$this->add_control(
			'content_text_align',
			array(
				'label'     => esc_html__( 'Alignment', 'azo' ),
				'type'      => Controls_Manager::CHOOSE,
				'options'   => array(
					'left'   => array(
						'title' => esc_html__( 'Left', 'azo' ),
						'icon'  => 'fa fa-align-left',
					),
					'center' => array(
						'title' => esc_html__( 'Center', 'azo' ),
						'icon'  => 'fa fa-align-center',
					),
					'right'  => array(
						'title' => esc_html__( 'Right', 'azo' ),
						'icon'  => 'fa fa-align-right',
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .widget-blog-post .entry-wrapper, .blog-article-sum, .info-item-blog' => 'text-align:{{VALUE}}',
				),
				'default'   => 'left',
			)
		);

		$this->add_control(
			'date',
			array(
				'label'        => esc_html__( 'Show Date', 'azo' ),
				'type'         => Controls_Manager::SWITCHER,
				'separator'    => 'before',
				'default'      => 'yes',
				'on'           => esc_html__( 'No', 'azo' ),
				'off'          => esc_html__( 'Yes', 'azo' ),
				'return_value' => 'yes',
			)
		);

		// Show excerpt.
		$this->add_control(
			'excerpt_option',
			array(
				'label'        => esc_html__( 'Show Excerpt', 'azo' ),
				'type'         => Controls_Manager::SWITCHER,
				'default'      => 'yes',
				'label_on'     => esc_html__( 'No', 'azo' ),
				'label_off'    => esc_html__( 'Yes', 'azo' ),
				'return_value' => 'yes',
			)
		);

		$this->add_control(
			'lenght-text',
			array(
				'label'     => esc_html__( 'Lenght text', 'azo' ),
				'type'      => Controls_Manager::NUMBER,
				'default'   => 12,
				'condition' => array(
					'excerpt_option' => 'yes',
				),
			)
		);

		$this->add_control(
			'button_name',
			array(
				'label'   => esc_html__( 'Button name', 'azo' ),
				'type'    => Controls_Manager::TEXT,
				'default' => esc_html__( 'Continue Reading', 'azo' ),
			)
		);

		$this->add_control(
			'icon_button',
			array(
				'label'            => esc_html__( 'Icon Button', 'azo' ),
				'type'             => Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'skin'             => 'inline',
				'label_block'      => false,
			)
		);

		$this->add_responsive_control(
			'columns',
			array(
				'label'          => esc_html__( 'Columns', 'azo' ),
				'type'           => Controls_Manager::SELECT,
				'default'        => '3',
				'tablet_default' => '2',
				'mobile_default' => '1',
				'options'        => array(
					'1' => '1',
					'2' => '2',
					'3' => '3',
					'4' => '4',
					'5' => '5',
				),
				'condition'      => array(
					'blog_preset' => array(
						'blogpreset2',
						'blogpreset4',
					),
				),
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Register query section controls
	 */
	protected function register_query_section_controls() {
		$this->start_controls_section(
			'section_query',
			array(
				'label' => esc_html__( 'Query', 'azo' ),
			)
		);

		$this->add_control(
			'categories',
			array(
				'label'    => esc_html__( 'Categories', 'azo' ),
				'type'     => Controls_Manager::SELECT2,
				'options'  => azo_get_narrow_data( 'term', 'category' ),
				'multiple' => true,
			)
		);

		$this->add_control(
			'posts_per_page',
			array(
				'label'   => esc_html__( 'Posts Per Page', 'azo' ),
				'type'    => Controls_Manager::NUMBER,
				'default' => 6,
			)
		);

		$this->add_control(
			'orderby',
			array(
				'label'   => esc_html__( 'Order By', 'azo' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'post_date',
				'options' => array(
					'post_date'  => esc_html__( 'Date', 'azo' ),
					'post_title' => esc_html__( 'Title', 'azo' ),
					'rand'       => esc_html__( 'Random', 'azo' ),
				),
			)
		);

		$this->add_control(
			'order',
			array(
				'label'   => esc_html__( 'Order', 'azo' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'desc',
				'options' => array(
					'asc'  => esc_html__( 'ASC', 'azo' ),
					'desc' => esc_html__( 'DESC', 'azo' ),
				),
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Register pagination section controls
	 */
	protected function register_pagination_section_controls() {
		$this->start_controls_section(
			'section_pagination',
			array(
				'label' => esc_html__( 'Pagination', 'azo' ),
			)
		);

		$this->add_control(
			'show_pagination',
			array(
				'label'        => esc_html__( 'Show Pagination', 'azo' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', 'azo' ),
				'label_off'    => esc_html__( 'Hide', 'azo' ),
				'return_value' => 'yes',
				'default'      => 'no',
			)
		);

		$this->add_control(
			'pagination_text_align',
			array(
				'label'     => esc_html__( 'Alignment', 'azo' ),
				'type'      => Controls_Manager::CHOOSE,
				'options'   => array(
					'flex-start' => array(
						'title' => esc_html__( 'Left', 'azo' ),
						'icon'  => 'fa fa-align-left',
					),
					'center'     => array(
						'title' => esc_html__( 'Center', 'azo' ),
						'icon'  => 'fa fa-align-center',
					),
					'flex-end'   => array(
						'title' => esc_html__( 'Right', 'azo' ),
						'icon'  => 'fa fa-align-right',
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .azo-pagination' => 'justify-content:{{VALUE}}',
				),
				'default'   => 'center',
				'condition' => array(
					'show_pagination' => 'yes',
				),
			)
		);

		$this->add_responsive_control(
			'pagination_spacing',
			array(
				'label'          => esc_html__( 'Spacing', 'azo' ),
				'type'           => Controls_Manager::DIMENSIONS,
				'size_units'     => array( 'px', '%', 'em' ),
				'default'        => array(
					'top'      => '30',
					'right'    => '0',
					'bottom'   => '0',
					'left'     => '0',
					'unit'     => 'px',
					'isLinked' => false,
				),
				'tablet_default' => array(
					'top'      => '20',
					'right'    => '0',
					'bottom'   => '0',
					'left'     => '0',
					'unit'     => 'px',
					'isLinked' => false,
				),
				'mobile_default' => array(
					'top'      => '15',
					'right'    => '0',
					'bottom'   => '0',
					'left'     => '0',
					'unit'     => 'px',
					'isLinked' => false,
				),
				'selectors'      => array(
					'{{WRAPPER}} .azo-pagination' => 'padding:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				),
				'condition'      => array(
					'show_pagination' => 'yes',
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'item_blog',
			array(
				'label' => esc_html__( 'Item Blog', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_responsive_control(
			'img_height',
			array(
				'label'     => __( 'Height Image', 'azo' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min' => 1,
						'max' => 1000,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .widget-blog-post .item-blog .entry-thumbnail img, .widget-blog-post2 .item-blog .entry-thumbnail img, .blog .cover-image img, .blogpreset3 .blog-preset-3 .item-blog .cover-image img' => 'height: {{SIZE}}{{UNIT}}; width: 100%;',
					'{{WRAPPER}} .blog4 .blog-preset-4 .item-blog .cover-image img' => 'height: {{SIZE}}{{UNIT}}; width: 100%;',
				),
				'separator' => 'before',
			)
		);

		$this->add_control(
			'br-image',
			array(
				'label'      => esc_html__( 'Border Radius Images', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .item-blog .entry-thumbnail img' => 'border-radius:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
					'{{WRAPPER}} .item-blog' => 'border-radius:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				),
			)
		);

		$this->add_responsive_control(
			'padding_button_item',
			array(
				'label'     => __( 'Spacing Bottom', 'azo' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min' => 0,
						'max' => 100,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .widget-blog-post .item-blog, .widget-blog-post2 .item-blog, .blog-article, .blog3 .blog-preset-3, .blog4 .blog-preset-4' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->add_responsive_control(
			'padding_info_blog',
			array(
				'label'      => esc_html__( 'Padding Info', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .info-item-blog, .blog-article-sum' => 'padding:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				),
			)
		);

		$this->start_controls_tabs(
			'style_info_hover'
		);

		$this->start_controls_tab(
			'tab_title_normal',
			array(
				'label' => esc_html__( 'Nomal', 'azo' ),
			)
		);

		$this->add_control(
			'bgcolor_info',
			array(
				'label'     => esc_html__( 'Background', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .item-blog' => 'background-color: {{VALUE}}',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'label'    => esc_html__( 'Box Shadow', 'azo' ),
				'name'     => 'box_shadow_blog_slider',
				'selector' => '{{WRAPPER}} .item-blog',
			)
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tab_title_hover',
			array(
				'label' => esc_html__( 'Hover', 'azo' ),
			)
		);

		$this->add_control(
			'bgcolor_info_hover',
			array(
				'label'     => esc_html__( 'Background', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .item-blog:hover' => 'background-color: {{VALUE}}',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			array(
				'label'    => esc_html__( 'Box Shadow', 'azo' ),
				'name'     => 'box_shadow_blog_slider_hover',
				'selector' => '{{WRAPPER}} .item-blog:hover',
			)
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->end_controls_section();

		$this->start_controls_section(
			'section_date',
			array(
				'label'     => esc_html__( 'Date', 'azo' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => array(
					'date' => 'yes',
				),
			)
		);

		$this->add_control(
			'date_color',
			array(
				'label'     => esc_html__( 'Date Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#222222',
				'selectors' => array(
					'{{WRAPPER}} .entry-thumbnail .azo-date-time'  => 'color: {{VALUE}}',
					'{{WRAPPER}} .entry-thumbnail .azo-date-time span.space'  => 'background-color: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'date_bg_color',
			array(
				'label'     => esc_html__( 'Date Background Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffb700',
				'selectors' => array(
					'{{WRAPPER}} .entry-thumbnail .azo-date-time'  => 'background-color: {{VALUE}}',
				),
			)
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			array(
				'label'    => esc_html__( 'Typography Date', 'azo' ),
				'name'     => 'date_typography',
				'selector' => '{{WRAPPER}} .entry-thumbnail .azo-date-time .day',
			)
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			array(
				'label'    => esc_html__( 'Typography Month', 'azo' ),
				'name'     => 'month_typography',
				'selector' => '{{WRAPPER}} .entry-thumbnail .azo-date-time .month',
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'header-blog-style',
			array(
				'label' => esc_html__( 'Header Info', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'header-info-typo',
				'label'    => esc_html__( 'Typography Info', 'azo' ),
				'selector' => '{{WRAPPER}} .info-blog-change .if-cat a',
			)
		);

		$this->add_control(
			'info-color',
			array(
				'label'     => esc_html__( 'Info Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .info-blog-change .if-cat a' => 'color: {{VALUE}}',
				),
				'default'   => '#e76700',
			)
		);

		$this->add_control(
			'info-color-hover',
			array(
				'label'     => esc_html__( 'Info Color Hover', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .info-blog-change .if-cat a:hover' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_responsive_control(
			'category-margin',
			array(
				'label'      => esc_html__( 'Margin', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .info-blog-change' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'title-blog-style',
			array(
				'label' => esc_html__( 'Title', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'title_typo',
				'label'    => esc_html__( 'Typography', 'azo' ),
				'selector' => '{{WRAPPER}} .blog-entry-title, {{WRAPPER}} .widget-blog-post .entry-title, {{WRAPPER}} .blog-article-sum .entry-title, {{WRAPPER}} .blogpreset3 .blog-preset-3 .item-blog .entry-title.blog-entry-title, {{WRAPPER}} .blog4 .blog-preset-4 .item-blog .entry-title.blog-entry-title',
			)
		);

		$this->add_control(
			'title_color',
			array(
				'label'     => esc_html__( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .blog-entry-title a, .widget-blog-post .entry-title a, .blog-article-sum .entry-title a, .blogpreset3 .blog-preset-3 .item-blog .entry-title.blog-entry-title a' => 'color: {{VALUE}}',
				),
				'default'   => '#222222',
			)
		);

		$this->add_control(
			'title_color_hover',
			array(
				'label'     => esc_html__( 'Hover', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .blog-entry-title a:hover, .widget-blog-post .entry-title a:hover, .blog-article-sum .entry-title a:hover, .blogpreset3 .blog-preset-3 .item-blog .entry-title.blog-entry-title a:hover' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_responsive_control(
			'title-margin',
			array(
				'label'      => esc_html__( 'Margin', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .blog-entry-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					'{{WRAPPER}} .blog4 .blog-preset-4 .item-blog .entry-title.blog-entry-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'description_style',
			array(
				'label'     => esc_html__( 'Description', 'azo' ),
				'tab'       => Controls_Manager::TAB_STYLE,
				'condition' => array(
					'excerpt_option' => 'yes',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'description_typo',
				'label'    => esc_html__( 'Typography', 'azo' ),
				'selector' => '{{WRAPPER}} .entry-content',
			)
		);

		$this->add_control(
			'color-description',
			array(
				'label'     => esc_html__( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .entry-content ' => 'color: {{VALUE}}',
				),
				'default'   => '#858585',
			)
		);

		$this->add_responsive_control(
			'description-margin',
			array(
				'label'      => esc_html__( 'Margin', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .entry-content' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style',
			array(
				'label' => esc_html__( 'Readmore', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'readmore_typo',
				'label'    => esc_html__( 'Typography Text Button', 'azo' ),
				'selector' => '{{WRAPPER}} .button-read-more .button-text',
			)
		);

		$this->add_responsive_control(
			'icon_typo',
			array(
				'label'     => __( 'Icon Typography', 'azo' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min' => 1,
						'max' => 300,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .btnblog-readmore .button-icon-svg svg' => 'width: {{SIZE}}{{UNIT}}; height:auto;',
					'{{WRAPPER}} .btnblog-readmore .button-icon' => 'font-size: {{SIZE}}{{UNIT}};',
				),
				'separator' => 'before',
			)
		);

		$this->add_responsive_control(
			'icon_padding',
			array(
				'label'     => __( 'Icon Space', 'azo' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min' => 1,
						'max' => 50,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .btnblog-readmore .button-icon-svg svg' => 'margin-left: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .btnblog-readmore .button-icon' => 'margin-left: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->add_control(
			'readmore_color',
			array(
				'label'      => esc_html__( 'Color', 'azo' ),
				'type'       => Controls_Manager::COLOR,
				'default'    => '#222222',
				'selectors'  => array(
					'{{WRAPPER}} .button-read-more .button-text' => 'color: {{VALUE}}',
					'{{WRAPPER}} .btnblog-readmore .button-icon' => 'color: {{VALUE}}',
					'{{WRAPPER}} .btnblog-readmore .button-icon-svg svg' => 'fill: {{VALUE}}',
				),
				'separators' => 'before',
			)
		);

		$this->add_control(
			'readmore_color_hover',
			array(
				'label'      => esc_html__( 'Color Hover', 'azo' ),
				'type'       => Controls_Manager::COLOR,
				'default'    => '#ffb700',
				'selectors'  => array(
					'{{WRAPPER}} .button-read-more:hover .button-text' => 'color: {{VALUE}}',
					'{{WRAPPER}} .btnblog-readmore:hover .button-icon' => 'color: {{VALUE}}',
					'{{WRAPPER}} .btnblog-readmore:hover .button-icon-svg svg' => 'fill: {{VALUE}}',
				),
				'separators' => 'before',
			)
		);

		$this->add_responsive_control(
			'readmore-margin',
			array(
				'label'      => esc_html__( 'Margin', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%' ),
				'selectors'  => array(
					'{{WRAPPER}} .button-read-more' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 */
	protected function render() {
		$settings    = $this->get_settings_for_display();
		$categories  = $settings['categories'];
		$lenght      = $settings['lenght-text'];
		$buttonname  = $settings['button_name'];
		$button_icon = $settings['icon_button'];
		$classes     = 'wd-blog ht-grid ht-grid-' . $settings['columns'] . ' ht-grid-tablet-' . $settings['columns_tablet'] . ' ht-grid-mobile-' . $settings['columns_mobile'];
		$paged       = get_query_var( 'paged' ) ? intval( get_query_var( 'paged' ) ) : 1;
		$args        = array(
			'post_type'      => 'post',
			'post_status'    => 'publish',
			'posts_per_page' => $settings['posts_per_page'],
			'orderby'        => $settings['orderby'],
			'order'          => $settings['order'],
			'paged'          => $paged,
		);
		if ( ! empty( $categories ) ) {
			$args['cat'] = $categories;
		}

		$the_query  = new \WP_Query( $args );
		$total_page = $the_query->max_num_pages;

		$blog_preset = $settings['blog_preset'];
		$excerpt     = $settings['excerpt_option'];

		if ( 'blogpreset1' === $blog_preset ) {
			?>
			<div class="blog default-blog">
				<?php
				while ( $the_query->have_posts() ) :
					$the_query->the_post();
					?>
					<div class="blog-article" <?php azo_schema_markup( 'blog' ); ?>>
						<article id="post-<?php the_ID(); ?>" <?php post_class(); ?> <?php azo_schema_markup( 'blog_list' ); ?>>
							<div itemprop="mainEntityOfPage">
								<?php if ( has_post_thumbnail() && false !== get_theme_mod( 'image_blog', true ) ) { ?>
									<div class="entry-image">
										<?php azo_post_format(); ?>
									</div>
								<?php } ?>

								<div class="blog-article-sum">
									<div class="info-blog-change">
										<span class="if-item if-cat">
											<?php echo wp_kses( azo_blog_categories(), azo_get_allowed_html() ); ?>
										</span>
									</div>

									<div class="blog-article-header">
										<header class="entry-header">
											<?php the_title( '<h3 class="entry-title blog-title"><a href="' . esc_url( get_permalink() ) . '" rel="bookmark">', '</a></h3>' ); ?>
										</header>
									</div>

									<div class="entry-summary"><?php azo_seo_data(); ?></div>

									<?php if ( 'yes' === $excerpt ) { ?>
										<div class="entry-content" <?php azo_schema_markup( 'post_content' ); ?>>
											<?php echo esc_html( wp_trim_words( get_the_content( get_the_ID() ), $lenght, '' ) ); ?>
										</div>
									<?php } ?>

									<?php if ( ! empty( $buttonname ) ) { ?>
										<div class="button-read-more">
											<a href="<?php the_permalink(); ?>" class="btnblog-readmore">
												<span class="button-align">
													<span class="button-text"><?php echo esc_html( $buttonname ); ?></span>
													<?php
													if ( ! empty( $button_icon['value'] ) ) :
														if ( is_string( $button_icon['value'] ) ) :
															?>
															<span class="button-icon <?php echo esc_attr( $button_icon['value'] ); ?>"></span>
														<?php else : ?>
															<span class="button-icon button-icon-svg">
																<?php Icons_Manager::render_icon( $button_icon, '' ); ?>
															</span>
															<?php
														endif;
													endif;
													?>
												</span>
											</a>
										</div>
									<?php } ?>
								</div>
							</div>
						</article>
					</div>
					<?php
				endwhile;

				if ( 'yes' === $settings['show_pagination'] ) {
					azo_pagination( $total_page );
				}
				?>
			</div>
			<?php
		} elseif ( 'blogpreset2' === $blog_preset ) {
			?>
			<div class="<?php echo esc_attr( $classes ); ?> widget-blog-post">
				<?php
				while ( $the_query->have_posts() ) :
					$the_query->the_post();
					?>
					<div class="ht-grid-item widget-blog-post-<?php echo esc_attr( $settings['layout'] ); ?>" <?php azo_schema_markup( 'article' ); ?> >
						<div class="item-blog entry-wrapper">
							<div class="entry-thumbnail">
								<a href="<?php the_permalink(); ?>">
									<?php
									if ( has_post_thumbnail() ) {
										the_post_thumbnail( 'full' );
									}
									?>
									<?php
									if ( 'yes' === $settings['date'] ) {
										?>
										<div class="azo-date-time">
											<?php echo esc_html( azo_date_format() ); ?>
										</div>
									<?php } ?>
								</a>
							</div>

							<div class="info-item-blog">
								<div class="info-blog-change">
									<span class="if-item if-cat">
										<?php echo wp_kses( azo_blog_categories(), azo_get_allowed_html() ); ?>
									</span>
								</div>

								<h6 class="entry-title blog-entry-title">
									<a href="<?php the_permalink(); ?>"><?php the_title(); ?></a>
								</h6>

								<?php if ( 'yes' === $excerpt ) { ?>
									<div class="entry-content">
										<?php echo esc_html( wp_trim_words( get_the_content( get_the_ID() ), $lenght, '' ) ); ?>
									</div>
								<?php } ?>

								<?php if ( ! empty( $buttonname ) ) { ?>
									<div class="button-read-more">
										<a href="<?php the_permalink(); ?>" class="btnblog-readmore">
											<span class="button-align">
												<span class="button-text"><?php echo esc_html( $buttonname ); ?></span>
												<?php
												if ( ! empty( $button_icon['value'] ) ) :
													if ( is_string( $button_icon['value'] ) ) :
														?>
														<span class="button-icon <?php echo esc_attr( $button_icon['value'] ); ?>"></span>
													<?php else : ?>
														<span class="button-icon button-icon-svg">
															<?php Icons_Manager::render_icon( $button_icon, '' ); ?>
														</span>
														<?php
													endif;
												endif;
												?>
											</span>
										</a>
									</div>
								<?php } ?>
							</div>
						</div>
					</div>
					<?php
				endwhile;
				?>
			</div>
			<?php
			if ( 'yes' === $settings['show_pagination'] ) {
				azo_pagination( $total_page );
			}
		} elseif ( 'blogpreset3' === $blog_preset ) {
			?>
			<div class="blog4">
				<?php
				while ( $the_query->have_posts() ) :
					$the_query->the_post();
					?>
					<div class="blog-article blog-preset-4" <?php azo_schema_markup( 'blog' ); ?>>
						<div class="item-blog entry-wrapper">
							<?php if ( has_post_thumbnail() && false !== get_theme_mod( 'image_blog', true ) ) { ?>
								<div class="entry-thumbnail">
									<a href="<?php the_permalink(); ?>">
										<?php
										if ( has_post_thumbnail() ) {
											the_post_thumbnail( 'full' );
										}
										?>
										<?php
										if ( 'yes' === $settings['date'] ) {
											?>
											<div class="azo-date-time">
												<?php echo esc_html( azo_date_format() ); ?>
											</div>
										<?php } ?>
									</a>
								</div>
							<?php } ?>

							<div class="info-item-blog">
								<div class="info-blog-change">
									<span class="if-item if-cat">
										<?php echo wp_kses( azo_blog_categories(), azo_get_allowed_html() ); ?>
									</span>
								</div>

								<?php the_title( '<h6 class="entry-title blog-entry-title"><a href="' . esc_url( get_permalink() ) . '" rel="bookmark">', '</a></h6>' ); ?>

								<?php if ( 'yes' === $excerpt ) { ?>
									<div class="entry-content">
										<?php echo esc_html( wp_trim_words( get_the_content( get_the_ID() ), $lenght, '' ) ); ?>
									</div>
								<?php } ?>

								<?php if ( ! empty( $buttonname ) ) { ?>
									<div class="button-read-more">
										<a href="<?php the_permalink(); ?>" class="btnblog-readmore">
											<span class="button-align">
												<span class="button-text"><?php echo esc_html( $buttonname ); ?></span>
												<?php
												if ( ! empty( $button_icon['value'] ) ) :
													if ( is_string( $button_icon['value'] ) ) :
														?>
														<span class="button-icon <?php echo esc_attr( $button_icon['value'] ); ?>"></span>
													<?php else : ?>
														<span class="button-icon button-icon-svg">
															<?php Icons_Manager::render_icon( $button_icon, '' ); ?>
														</span>
														<?php
													endif;
												endif;
												?>
											</span>
										</a>
									</div>
								<?php } ?>
							</div>
						</div>
					</div>
					<?php
				endwhile;

				if ( 'yes' === $settings['show_pagination'] ) {
					azo_pagination( $total_page );
				}
				?>
			</div>
			<?php
		}
	}
}

Plugin::instance()->widgets_manager->register_widget_type( new Azo_Blog_Posts() );
