<?php
/**
 * Azo banner widget
 *
 * @package azo
 */

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Azo banner widget.
 *
 * Azo widget that displays an image, a headline, a text and a button.
 *
 * @since 1.0.0
 */
class Azo_Banner extends Widget_Base {
	/**
	 * Get widget name.
	 *
	 * Retrieve banner widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'azo-banner';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve banner widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Banner Azo', 'azo' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve banner widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-image-box';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the icon widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return array( 'azo-theme' );
	}

	/**
	 * Register banner widget controls.
	 *
	 * Add different input fields to allow the user to change and customize the widget settings
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() { //phpcs:ignore
		$this->start_controls_section(
			'section_category',
			array(
				'label' => esc_html__( 'Banner', 'azo' ),
			)
		);

		$this->add_control(
			'image',
			array(
				'label'   => esc_html__( 'Choose Image', 'azo' ),
				'type'    => Controls_Manager::MEDIA,
				'dynamic' => array(
					'active' => true,
				),
				'default' => array(
					'url' => Utils::get_placeholder_image_src(),
				),
			)
		);

		$this->add_group_control(
			Group_Control_Image_Size::get_type(),
			array(
				'name'      => 'thumbnail',
				'default'   => 'full',
				'separator' => 'none',
			)
		);

		$this->add_control(
			'sub_title',
			array(
				'label'       => esc_html__( 'Sub Title', 'azo' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => esc_html__( 'This is Sub Title', 'azo' ),
				'placeholder' => esc_html__( 'Enter your text sub title', 'azo' ),
				'label_block' => true,
			)
		);

		$this->add_control(
			'title_text',
			array(
				'label'       => esc_html__( 'Title', 'azo' ),
				'type'        => Controls_Manager::TEXT,
				'dynamic'     => array(
					'active' => true,
				),
				'default'     => esc_html__( 'This is the heading', 'azo' ),
				'placeholder' => esc_html__( 'Enter your text title', 'azo' ),
				'label_block' => true,
			)
		);

		$this->add_control(
			'description',
			array(
				'label'   => esc_html__( 'Description', 'azo' ),
				'type'    => Controls_Manager::TEXTAREA,
				'default' => esc_html__( '102 Iteam', 'azo' ),
			)
		);

		$this->add_control(
			'button_text',
			array(
				'label'       => esc_html__( 'Button name', 'azo' ),
				'type'        => Controls_Manager::TEXT,
				'dynamic'     => array(
					'active' => true,
				),
				'default'     => esc_html__( 'Click here', 'azo' ),
				'placeholder' => esc_html__( 'Enter your button title', 'azo' ),
				'label_block' => true,
			)
		);

		$this->add_control(
			'style_button',
			array(
				'label'   => esc_html__( 'Choose style Button', 'azo' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'style1' => esc_html__( 'Style 1', 'azo' ),
					'style2' => esc_html__( 'Style 2', 'azo' ),
				),
				'default' => 'style1',
			)
		);

		$this->add_control(
			'link',
			array(
				'label'       => esc_html__( 'Link to', 'azo' ),
				'type'        => Controls_Manager::URL,
				'dynamic'     => array(
					'active' => true,
				),
				'placeholder' => esc_html__( 'https://your-link.com', 'azo' ),
				'separator'   => 'before',
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_content',
			array(
				'label' => esc_html__( 'Content', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_responsive_control(
			'padding_content',
			array(
				'label'      => esc_html__( 'Padding Content', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .azo-banner-content' => 'padding:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				),
			)
		);

		$this->add_control(
			'br-image',
			array(
				'label'      => esc_html__( 'Border Radius', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .azo-banner-image' => 'border-radius:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				),
			)
		);

		$this->add_responsive_control(
			'height_img_banner',
			array(
				'label'     => esc_html__( 'Height Image', 'azo' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'px' => array(
						'min' => 0,
						'max' => 1000,
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .azo-banner-image img' => 'height: {{SIZE}}{{UNIT}}',
				),
			)
		);

		$this->add_responsive_control(
			'text_align',
			array(
				'label'     => esc_html__( 'Alignment', 'azo' ),
				'type'      => Controls_Manager::CHOOSE,
				'options'   => array(
					'flex-start' => array(
						'title' => esc_html__( 'Left', 'azo' ),
						'icon'  => 'fa fa-align-left',
					),
					'center'     => array(
						'title' => esc_html__( 'Center', 'azo' ),
						'icon'  => 'fa fa-align-center',
					),
					'flex-end'   => array(
						'title' => esc_html__( 'Right', 'azo' ),
						'icon'  => 'fa fa-align-right',
					),
				),
				'default'   => 'flex-start',
				'selectors' => array(
					'{{WRAPPER}} .azo-banner-content' => 'align-items:{{VALUE}};',
				),
			)
		);

		$this->add_responsive_control(
			'vetical_align',
			array(
				'label'     => esc_html__( 'Vertical', 'azo' ),
				'type'      => Controls_Manager::CHOOSE,
				'options'   => array(
					'flex-start' => array(
						'title' => esc_html__( 'Top', 'azo' ),
						'icon'  => 'fa fa-chevron-up',
					),
					'center'     => array(
						'title' => esc_html__( 'Center', 'azo' ),
						'icon'  => 'fa fa-align-justify',
					),
					'flex-end'   => array(
						'title' => esc_html__( 'Bottom', 'azo' ),
						'icon'  => 'fa fa-chevron-down',
					),
				),
				'selectors' => array(
					'{{WRAPPER}} .azo-banner-content' => 'justify-content:{{VALUE}};',
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_subtitle',
			array(
				'label' => esc_html__( 'Sub Title', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'sub-typography',
				'selector' => '{{WRAPPER}} .azo-banner-subtitle',
			)
		);

		$this->add_control(
			'color_subtitle',
			array(
				'label'     => esc_html__( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-banner-subtitle' => 'color: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'bgcolor_subtitle',
			array(
				'label'     => esc_html__( 'Background', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-banner-subtitle' => 'background-color: {{VALUE}}',
				),
			)
		);

		$this->add_responsive_control(
			'subtitle_padding',
			array(
				'label'      => esc_html__( 'Padding', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .azo-banner-subtitle' => 'padding:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				),
			)
		);

		$this->add_control(
			'br-subtitle',
			array(
				'label'      => esc_html__( 'Border Radius', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .azo-banner-subtitle' => 'border-radius:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				),
			)
		);

		$this->add_responsive_control(
			'margin_subtitle',
			array(
				'label'      => esc_html__( 'Margin', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .azo-banner-subtitle' => 'margin:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_title',
			array(
				'label' => esc_html__( 'Title', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'title-typography',
				'selector' => '{{WRAPPER}} .azo-banner-title',
			)
		);

		$this->start_controls_tabs(
			'style_title_tab'
		);

		$this->start_controls_tab(
			'tab_title_normal',
			array(
				'label' => esc_html__( 'Normal', 'azo' ),
			)
		);

		$this->add_control(
			'color_title',
			array(
				'label'     => esc_html__( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-banner-title a' => 'color: {{VALUE}}',
				),
			)
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tab_title_hover',
			array(
				'label' => esc_html__( 'Hover', 'azo' ),
			)
		);

		$this->add_control(
			'color_title_hover',
			array(
				'label'     => esc_html__( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-banner-wrapper:hover .azo-banner-title a' => 'color: {{VALUE}}',
				),
			)
		);

		$this->end_controls_tabs();

		$this->add_responsive_control(
			'margin_title',
			array(
				'label'      => esc_html__( 'Margin', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .azo-banner-title' => 'margin:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_description',
			array(
				'label' => esc_html__( 'Description', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'description_typography',
				'selector' => '{{WRAPPER}} .azo-banner-description',
			)
		);

		$this->add_control(
			'description_color',
			array(
				'label'     => esc_html__( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-banner-description' => 'color:{{VALUE}};',
				),
			)
		);

		$this->add_control(
			'bgcolor_description',
			array(
				'label'     => esc_html__( 'Background', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => array(
					'{{WRAPPER}} .azo-banner-description' => 'background-color: {{VALUE}}',
				),
			)
		);

		$this->add_responsive_control(
			'description_padding',
			array(
				'label'      => esc_html__( 'Padding', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .azo-banner-description' => 'padding:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				),
			)
		);

		$this->add_control(
			'br-description',
			array(
				'label'      => esc_html__( 'Border Radius', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .azo-banner-description' => 'border-radius:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				),
			)
		);

		$this->add_responsive_control(
			'description_margin',
			array(
				'label'      => esc_html__( 'Margin', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .azo-banner-description' => 'margin:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_button_banner',
			array(
				'label' => esc_html__( 'Button', 'azo' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			array(
				'name'     => 'button_typography',
				'selector' => '{{WRAPPER}} .azo-banner-button',
			)
		);

		$this->start_controls_tabs(
			'style_button_tab'
		);

		$this->start_controls_tab(
			'style_button_banner_nomal',
			array(
				'label' => esc_html__( 'Nomal', 'azo' ),
			)
		);

		$this->add_control(
			'button_color',
			array(
				'label'     => esc_html__( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#fff',
				'selectors' => array(
					'{{WRAPPER}} .azo-banner-button' => 'color: {{VALUE}}',
					'{{WRAPPER}} .azo-banner-button.style2:after' => 'background-color: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'button_bg_color',
			array(
				'label'     => esc_html__( 'Background Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#000',
				'selectors' => array(
					'{{WRAPPER}} .azo-banner-button' => 'background-color: {{VALUE}}',
				),
				'condition' => array(
					'style_button' => 'style1',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			array(
				'name'        => 'border',
				'placeholder' => '1px',
				'default'     => '1px',
				'selector'    => '{{WRAPPER}} .azo-banner-button',
				'condition'   => array(
					'style_button' => 'style1',
				),
			)
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'style_button_banner_hover',
			array(
				'label' => esc_html__( 'Hover', 'azo' ),
			)
		);

		$this->add_control(
			'button_color_hover',
			array(
				'label'     => esc_html__( 'Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => array(
					'{{WRAPPER}} .azo-banner-wrapper:hover .azo-banner-button' => 'color: {{VALUE}}',
					'{{WRAPPER}} .azo-banner-button.style2:before' => 'background-color: {{VALUE}}',
				),
			)
		);

		$this->add_control(
			'button_bg_color_hover',
			array(
				'label'     => esc_html__( 'Background Color', 'azo' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => array(
					'{{WRAPPER}} .azo-banner-wrapper:hover .azo-banner-button' => 'background-color: {{VALUE}}',
				),
				'condition' => array(
					'style_button' => 'style1',
				),
			)
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			array(
				'name'        => 'border-hover',
				'placeholder' => '1px',
				'default'     => '1px',
				'selector'    => '{{WRAPPER}} .azo-banner-wrapper:hover .azo-banner-button',
				'condition'   => array(
					'style_button' => 'style1',
				),
			)
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_responsive_control(
			'button_padding',
			array(
				'label'      => esc_html__( 'Padding', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .azo-banner-button' => 'padding:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				),
				'condition'  => array(
					'style_button' => 'style1',
				),
			)
		);

		$this->add_responsive_control(
			'button_margin',
			array(
				'label'      => esc_html__( 'Margin', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px', '%', 'em' ),
				'selectors'  => array(
					'{{WRAPPER}} .azo-banner-button' => 'margin:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				),
			)
		);

		$this->add_control(
			'border_radius',
			array(
				'label'      => __( 'Border Radius', 'azo' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => array( 'px' ),
				'selectors'  => array(
					'{{WRAPPER}} .azo-banner-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				),
				'condition'  => array(
					'style_button' => 'style1',
				),
			)
		);

		$this->end_controls_section();
	}

	/**
	 * Render banner widget output on the front end.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		$image       = $settings['image'];
		$descrip     = $settings['description'];
		$image_url   = $image['url'];
		$image_alt   = azo_img_alt( $image['id'], esc_attr__( 'Banner Image', 'azo' ) );
		$subtitle    = $settings['sub_title'];
		$title_text  = $settings['title_text'];
		$button_text = $settings['button_text'];

		$href = '#';
		$attr = '';

		if ( ! empty( $settings['link']['url'] ) ) {
			$href = $settings['link']['url'];

			if ( 'on' === $settings['link']['is_external'] ) {
				$attr .= ' target="_blank"';
			}

			if ( 'on' === $settings['link']['nofollow'] ) {
				$attr .= ' rel="nofollow"';
			}
		}
		?>
		<div class="azo-banner-wrapper">
			<a href="<?php echo esc_url( $href ); ?>" <?php echo wp_kses( $attr, azo_get_allowed_html() ); ?> class="azo-banner-link"></a>
			<div class="azo-banner-image">				
				<img src="<?php echo esc_url( $image_url ); ?>" alt="<?php echo esc_attr( $image_alt ); ?>">				
			</div>

			<div class="azo-banner-content">
				<?php if ( ! empty( $subtitle ) ) { ?>
					<span class="azo-banner-subtitle">
						<?php echo wp_kses( $subtitle, azo_get_allowed_html() ); ?>
					</span>
				<?php } ?>

				<?php if ( ! empty( $title_text ) ) { ?>
				<h3 class="azo-banner-title">
					<a href="<?php echo esc_url( $href ); ?>" <?php echo wp_kses( $attr, azo_get_allowed_html() ); ?>>
						<?php echo wp_kses( $title_text, azo_get_allowed_html() ); ?>
					</a>
				</h3>
				<?php } ?>

				<?php if ( ! empty( $descrip ) ) { ?>
				<p class="azo-banner-description">
					<?php echo wp_kses( $descrip, azo_get_allowed_html() ); ?>
				</p>
				<?php } ?>

				<?php if ( ! empty( $button_text ) && ( 'style1' === $settings['style_button'] ) ) { ?>
					<a href="<?php echo esc_url( $href ); ?>" <?php echo wp_kses( $attr, azo_get_allowed_html() ); ?> class="azo-banner-button style1">
						<?php echo esc_html( $button_text ); ?>
					</a>
				<?php } ?>

				<?php if ( ! empty( $button_text ) && ( 'style2' === $settings['style_button'] ) ) { ?>
					<a href="<?php echo esc_url( $href ); ?>" <?php echo wp_kses( $attr, azo_get_allowed_html() ); ?> class="azo-banner-button style2">
						<?php echo esc_html( $button_text ); ?>
					</a>
				<?php } ?>
			</div>
		</div>
		<?php
	}
}

Plugin::instance()->widgets_manager->register_widget_type( new Azo_Banner() );
