<?php
/**
 * Elementor Azo Products
 *
 * @package azo
 */

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

if ( ! defined( 'WC_PLUGIN_FILE' ) ) {
	return;
}

use Elementor\Controls_Manager;
use Elementor\Widget_Button;
use ElementorPro\Modules\QueryControl\Module;


/**
 * Azo Widget Products
 */
class Azo_Add_To_Cart extends Widget_Button {

	/**
	 * Get categories
	 */
	public function get_categories() {
		return array( 'azo-theme' );
	}

	/**
	 * Get name
	 */
	public function get_name() {
		return 'add-to-cart';
	}

	/**
	 * Get title
	 */
	public function get_title() {
		return esc_html__( 'Add To Cart', 'azo' );
	}

	/**
	 * Get Icon
	 */
	public function get_icon() {
		return 'eicon-product-add-to-cart';
	}

	protected function register_controls() { //phpcs:ignore
		$this->start_controls_section(
			'section_product',
			array(
				'label' => __( 'Product', 'azo' ),
			)
		);

		$this->add_control(
			'product_id',
			array(
				'label'    => __( 'Product', 'azo' ),
				'type'     => Controls_Manager::SELECT2,
				'options'  => azo_get_narrow_data( 'post', 'product' ),
				'multiple' => false,
			)
		);

		$this->end_controls_section();

		parent::register_controls();

		$this->update_control(
			'link',
			array(
				'type'    => Controls_Manager::HIDDEN,
				'default' => array(
					'url' => '',
				),
			)
		);

		$this->update_control(
			'text',
			array(
				'default'     => __( 'Add to Cart', 'azo' ),
				'placeholder' => __( 'Add to Cart', 'azo' ),
			)
		);

		$this->update_control(
			'selected_icon',
			array(
				'default' => array(
					'value'   => 'fas fa-shopping-cart',
					'library' => 'fa-solid',
				),
			)
		);
	}

	/**
	 * Render
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();

		if ( ! empty( $settings['product_id'] ) ) {
			$product_id = $settings['product_id'];
		} elseif ( wp_doing_ajax() ) {
			$product_id = $_POST['post_id']; //phpcs:ignore
		} else {
			$product_id = get_queried_object_id();
		}

		global $product;
		$product = wc_get_product( $product_id );

		$this->render_ajax_button( $product );
	}

	/**
	 * Render ajax button
	 *
	 * @param  [type] $product WC_Product $product.
	 */
	private function render_ajax_button( $product ) {
		$settings = $this->get_settings_for_display();

		if ( $product ) {
			if ( version_compare( WC()->version, '3.0.0', '>=' ) ) {
				$product_type = $product->get_type();
			} else {
				$product_type = $product->product_type;
			}

			$class = implode(
				' ',
				array_filter(
					array(
						'product_type_' . $product_type,
						$product->is_purchasable() && $product->is_in_stock() ? 'add_to_cart_button' : '',
						$product->supports( 'ajax_add_to_cart' ) ? 'ajax_add_to_cart' : '',
					)
				)
			);

			$this->add_render_attribute(
				'button',
				array(
					'rel'             => 'nofollow',
					'href'            => $product->add_to_cart_url(),
					'data-quantity'   => ( isset( $settings['quantity'] ) ? $settings['quantity'] : 1 ),
					'data-product_id' => $product->get_id(),
					'class'           => $class,
				)
			);

		} elseif ( current_user_can( 'manage_options' ) ) {
			$settings['text'] = __( 'Please set a valid product', 'azo' );
			$this->set_settings( $settings );
		}

		parent::render();
	}

	/**
	 * Render form button
	 *
	 * @param  [type] $product type.
	 */
	private function render_form_button( $product ) {
		if ( ! $product && current_user_can( 'manage_options' ) ) {
			echo esc_html__( 'Please set a valid product', 'azo' );

			return;
		}

		$text_callback = function() {
			ob_start();
			$this->render_text();

			return ob_get_clean();
		};

		ob_start();
		woocommerce_template_single_add_to_cart();
		$form = ob_get_clean();
		$form = str_replace( 'single_add_to_cart_button', 'single_add_to_cart_button elementor-button', $form );
		echo wp_kses( $form, azo_get_allowed_html() );
	}
}
Plugin::instance()->widgets_manager->register_widget_type( new Azo_Add_To_Cart() );
